<?php

namespace Mtc\ContentManager\Contracts;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Mtc\ContentManager\Factories\MediaFactory;
use Mtc\ContentManager\ImageSize;
use Mtc\ContentManager\Traits\ModelSortAndFilter;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

abstract class Media extends Model
{
    use HasFactory;
    use ModelSortAndFilter;

    protected static string $disk_url = '';

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'folder_id',
        'type',
        'src',
        'path',
        'legacy_structure',
        'alt_text',
        'title',
        'hex1',
        'hex2',
        'caption',
        'description',
        'uploaded_by',
        'external',
        'image_provider',
        'source_filename',
        'focal_point',
    ];

    /**
     * Hide attributes from json
     *
     * @var string[]
     */
    protected $hidden = [
        'created_at',
        'updated_at'
    ];

    protected $appends = [
        'original_url',
        'square_thumb_url',
        'resizable',
    ];

    /**
     * Cast attributes to types
     *
     * @var string[]
     */
    protected $casts = [
        'meta' => 'array',
        'external' => 'boolean',
    ];

    /**
     * Model Factory
     *
     * @return MediaFactory
     */
    protected static function newFactory()
    {
        return MediaFactory::new();
    }

    protected static function boot()
    {
        parent::boot();

        self::creating(static fn($self) => $self->upload_date = now()->format('Y-m'));
    }

    /**
     * @return HasMany
     */
    public function uses(): HasMany
    {
        return $this->hasMany(Config::get('media.media_use_model'), 'media_id');
    }

    public function folder(): BelongsTo
    {
        return $this->belongsTo(Config::get('media.media_folder_model'), 'folder_id');
    }

    /**
     * @return HasMany
     */
    public function tags(): HasMany
    {
        return $this->hasMany(Config::get('media.tag_model'), 'media_id');
    }

    /**
     * Add usage_count to query
     *
     * @param Builder $query
     */
    public function scopeWithUsageCount(Builder $query)
    {
        $query->addSelect([
            'usage_count' => function ($query) {
                $query->select(DB::raw('count(*)'))
                    ->from('media_uses')
                    ->whereColumn('media_id', 'media.id')
                    ->limit(1);
            }
        ]);
    }

    /**
     * Scope for getting current users uploaded files
     *
     * @param Builder $query
     * @return void
     */
    public function scopeMine(Builder $query)
    {
        $query->where('uploaded_by', auth()->id());
    }

    /**
     * check if media has resizable file type
     *
     * @return bool
     */
    public function resizableMime()
    {
        return $this->type === 'image';
    }

    /**
     * get resizable attribute
     *
     * @return bool
     */
    public function getResizableAttribute(): bool
    {
        return $this->resizableMime();
    }

    /**
     * square_thumb attribute
     *
     * @return string
     */
    public function getSquareThumbUrlAttribute(): string
    {
        if (!$this->resizableMime()) {
            return $this->getOriginalUrlAttribute();
        }

        if ($this->external) {
            return $this->externalUrl('square_thumb');
        }

        if (empty(self::$disk_url)) {
            self::$disk_url = rtrim(Storage::disk(Config::get('filesystems.default_media'))->url('a'), 'a');
        }

        $size = ImageSize::fromArray(Config::get('media.default_thumbnail_size'));
        return self::$disk_url . $size->pathOnDisk($this->src, $this->path, $this->legacy_structure);
    }

    /**
     * original_url attribute (url to the original size)
     *
     * @return string
     */
    public function getOriginalUrlAttribute(?bool $as_jpg = false): ?string
    {
        if ($this->external) {
            return $this->externalUrl('original');
        }

        if (empty(self::$disk_url)) {
            self::$disk_url = rtrim(Storage::disk(Config::get('filesystems.default_media'))->url('a'), 'a');
        }

        $filename = rawurlencode($this->src);
        if ($as_jpg) {
            $filename = str_replace(['.webp.webp', '.webp'], '.jpg', $this->src);
        }

        return self::$disk_url . rtrim($this->path, '/') . '/' . $filename;
    }

    /**
     * @return string
     */
    public function getOriginalFilePath(): string
    {
        return "$this->path/$this->src";
    }

    /**
     * @return string
     */
    public function thumbnailFilePath(): string
    {
        return ImageSize::fromArray(Config::get('media.default_thumbnail_size'))
            ->pathOnDisk($this->src, $this->path, $this->legacy_structure);
    }

    public function externalUrl(string $size): ?string
    {
        if ($this->external) {
            try {
                return app($this->image_provider)->url($this, $size);
            } catch (\Exception $exception) {
                Log::warning('Failed to retrieve external image url', [
                    $exception->getMessage(),
                    $this,
                ]);
            }
        }
        return null;
    }
}
