<?php

namespace Mtc\ContentManager\Contracts;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Support\Facades\Config;
use Mtc\ContentManager\Factories\MenuFactory;
use Mtc\ContentManager\Traits\ModelSortAndFilter;

abstract class Menu extends Model
{
    use HasFactory;
    use ModelSortAndFilter;

    /**
     * Table name
     *
     * @var string
     */
    protected $table = 'menus';

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'title',
        'slug',
        'description',
        'first_child_element_id',
        'data',
        'meta',
        'is_enabled',
    ];

    /**
     * Cast attributes to types
     *
     * @var string[]
     */
    protected $casts = [
        'data' => 'array',
        'meta' => 'array',
        'is_enabled' => 'boolean',
    ];

    /**
     * Boot model options
     */
    protected static function boot()
    {
        parent::boot();

        self::deleted(static fn (Menu $menu) => $menu->entries()->delete());
    }

    /**
     * Model factory
     *
     * @return MenuFactory
     */
    protected static function newFactory()
    {
        return MenuFactory::new();
    }

    /**
     * Relationship with menu entries
     *
     * @return HasMany
     */
    public function entries(): HasMany
    {
        return $this->hasMany(Config::get('pages.menu_entry_model'), 'menu_id');
    }

    /**
     * Relationship with menu entries
     *
     * @return HasMany
     */
    public function rootEntries(): HasMany
    {
        return $this->hasMany(Config::get('pages.menu_entry_model'), 'menu_id')
            ->orderBy('_lft')
            ->whereNull('parent_id');
    }

    public function activeRootEntries(): HasMany
    {
        return $this->rootEntries()->where('is_enabled', 1);
    }

    public function element(): BelongsTo
    {
        return $this->belongsTo(Config::get('pages.element_model'), 'first_child_element_id');
    }

    public function scopeActive(Builder $query): Builder
    {
        return $query->where('is_enabled', 1);
    }
}
