<?php

namespace Mtc\ContentManager\Contracts;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Support\Facades\Config;
use Mtc\ContentManager\Factories\MenuEntryFactory;
use Mtc\ContentManager\Traits\HasMedia;

abstract class MenuEntry extends Model
{
    use HasFactory;
    use HasMedia;

    /**
     * Table name
     *
     * @var string
     */
    protected $table = 'menu_entries';

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'menu_id',
        'element_id',
        'parent_id',
        '_lft',
        'name',
        'url',
        'content',
        'data',
        'is_enabled',
        'item_id',
        'item_type',
        'external',
        'css_class',
    ];

    /**
     * Cast attributes to types
     *
     * @var string[]
     */
    protected $casts = [
        'content' => 'array',
        'data' => 'array',
        'is_enabled' => 'boolean',
        'external' => 'boolean',
    ];

    /**
     * Model Factory
     *
     * @return MenuEntryFactory
     */
    protected static function newFactory()
    {
        return MenuEntryFactory::new();
    }

    /**
     * Relationship with menu
     *
     * @return BelongsTo
     */
    public function menu(): BelongsTo
    {
        return $this->belongsTo(Config::get('pages.menu_model'), 'menu_id');
    }

    /**
     * Item that defines the entry
     *
     * @return \Illuminate\Database\Eloquent\Relations\MorphTo
     */
    public function item()
    {
        return $this->morphTo();
    }

    public function element(): BelongsTo
    {
        return $this->belongsTo(Config::get('pages.element_model'));
    }

    /**
     * Relation to children.
     *
     * @return HasMany
     */
    public function children(): HasMany
    {
        return $this->hasMany(static::class, 'parent_id')
            ->orderBy('_lft');
    }

    public function activeChildren(): HasMany
    {
        return $this->children()->where('is_enabled', 1);
    }
}
