<?php

namespace Mtc\ContentManager\Providers;

use Illuminate\Console\Scheduling\Schedule;
use Illuminate\Contracts\Container\BindingResolutionException;
use Illuminate\Database\Eloquent\Relations\Relation;
use Illuminate\Foundation\Support\Providers\RouteServiceProvider;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\Route;
use Mtc\ContentManager\Console\Commands\DeleteOldPageVersions;
use Mtc\ContentManager\Console\Commands\DeleteOldTrashedPages;
use Mtc\ContentManager\Contracts\Comment;
use Mtc\ContentManager\Contracts\Content;
use Mtc\ContentManager\Contracts\ContentElement;
use Mtc\ContentManager\Contracts\ContentElementField;
use Mtc\ContentManager\Contracts\ElementRepositoryContract;
use Mtc\ContentManager\Contracts\GlobalContent;
use Mtc\ContentManager\Contracts\Menu;
use Mtc\ContentManager\Contracts\Template;
use Mtc\ContentManager\Contracts\PageModel;
use Mtc\ContentManager\Contracts\VersionModel;
use Mtc\ContentManager\Contracts\TemplateElement;
use Mtc\ContentManager\Models\ContentCategory;
use Mtc\ContentManager\Models\ContentTag;
use Mtc\ContentManager\Models\MenuEntry;

class ContentProvider extends RouteServiceProvider
{
    /**
     * Register the provider
     *
     * @return void
     */
    public function register(): void
    {
        $this->mergeConfigFrom(dirname(__DIR__, 2) . '/config/pages.php', 'pages');

        parent::register();
        if (config('pages.auto_load_migrations')) {
            $this->loadMigrationsFrom(__DIR__ . '/../../database/migrations/');
        }
        $this->loadTranslationsFrom(dirname(__DIR__, 2) . '/resources/lang', 'content');

        // Model Bindings
        $this->app->bind(PageModel::class, Config::get('pages.page_model'));
        $this->app->bind(Template::class, Config::get('pages.template_model'));
        $this->app->bind(TemplateElement::class, Config::get('pages.template_element_model'));
        $this->app->bind(VersionModel::class, Config::get('pages.version_model'));
        $this->app->bind(Content::class, Config::get('pages.content_model'));
        $this->app->bind(ContentElement::class, Config::get('pages.element_model'));
        $this->app->bind(ContentElementField::class, Config::get('pages.element_field_model'));
        $this->app->bind(GlobalContent::class, Config::get('pages.global_content_model'));
        $this->app->bind(Comment::class, Config::get('pages.comment_model'));
        $this->app->bind(Menu::class, Config::get('pages.menu_model'));
        $this->app->bind(MenuEntry::class, Config::get('pages.menu_entry_model'));
        $this->app->bind(ContentTag::class, Config::get('pages.tag_model'));
        $this->app->bind(ContentCategory::class, Config::get('pages.category_model'));
    }

    /**
     * Boot the provider
     *
     * @throws BindingResolutionException
     * @return void
     */
    public function boot(): void
    {
        $this->loadTranslationsFrom(__DIR__ . '/../../resources/lang', 'content');

        parent::boot();
        // Observers
        $this->app->make(PageModel::class)->observe(Config::get('pages.page_observer'));
        $this->app->make(GlobalContent::class)->observe(Config::get('pages.global_content_observer'));
        $this->app->make(Template::class)->observe(Config::get('pages.template_observer'));
        $this->app->make(MenuEntry::class)->observe(Config::get('pages.menu_entry_observer'));

        // Repositories
        $this->app->bind(ElementRepositoryContract::class, Config::get('pages.element_repository'));
        $this->app->bind('page_repository', Config::get('pages.page_repository'));
        $this->app->bind('comment-repository', Config::get('pages.comment_repository'));
        $this->app->bind('field_repository', Config::get('pages.field_repository'));
        $this->app->bind('menu_repository', Config::get('pages.menu_repository'));
        $this->app->bind('template_repository', Config::get('pages.template_repository'));

        // Relationships
        Relation::morphMap([
            'page' => Config::get('pages.page_model'),
            'element' => Config::get('pages.element_model'),
            'global-content' => Config::get('pages.global_content_model'),
            'version' => Config::get('pages.version_model'),
            'template' => Config::get('pages.template_model'),
            'menu' => Config::get('pages.menu_model'),
            'menu_entry' => Config::get('pages.menu_entry_model'),
            'comment' => Config::get('pages.comment_model'),
            'content' => Config::get('pages.content_model'),
            'version_content' => Config::get('pages.version_content_model'),
        ]);

        $this->commandsAndSchedule();

        $this->loadViewsFrom(dirname(__DIR__) . '/resources/views', 'content');

        $this->publishes([
            dirname(__DIR__, 2) . '/database/migrations' => $this->app->databasePath('migrations'),
        ], 'migrations');
    }

    /**
     * Define the "web" routes
     *
     * These routes all receive session state, CSRF protection, etc.
     *
     * @return void
     */
    public function map(): void
    {
        if (config('pages.load_routes') !== true) {
            return;
        }

        Route::prefix('api')
            ->middleware('api')
            ->group(dirname(__DIR__, 2) . '/routes/api.php');
    }

    /**
     * Register commands and schedule tasks
     *
     * @return void
     */
    public function commandsAndSchedule(): void
    {
        $this->commands([
            DeleteOldPageVersions::class,
            DeleteOldTrashedPages::class,
        ]);

        if (config('pages.schedule_tasks')) {
            $this->app->booted(function () {
                $schedule = $this->app->make(Schedule::class);
                $schedule->command(DeleteOldTrashedPages::class)->daily();
                $schedule->command(DeleteOldPageVersions::class)->daily();
            });
        }
    }
}
