<?php

namespace Mtc\ContentManager\Tests\Feature;

use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Carbon;
use Mtc\ContentManager\Models\ContentElement;
use Mtc\ContentManager\Models\ContentElementField;
use Mtc\ContentManager\Models\GlobalContent;
use Mtc\ContentManager\Models\Page;
use Mtc\ContentManager\Models\Template;
use Mtc\ContentManager\Models\Version;
use Mtc\ContentManager\PageStatus;
use Mtc\ContentManager\Tests\AsUser;
use Mtc\ContentManager\Tests\TestCase;

class PageControllerTest extends TestCase
{
    use RefreshDatabase;
    use AsUser;

    public function testIndex()
    {
        $this->asUser();

        Page::factory(5)->create();
        $response = $this->getJson(route('content.pages.index'));

        $response->assertStatus(200);
        $this->assertArrayHasKey('data', $response->json());
        $this->assertEquals(5, count($response->json('data')));
    }

    public function testIndexMyPages()
    {
        Page::factory(3)->create();
        $page = Page::factory()->create();

        $this->asUser();
        $page->users()->attach([$this->user->id]);
        $response = $this->getJson(route('content.pages.index', ['my-content' => true]));

        $response->assertStatus(200);
        $this->assertEquals(1, count($response->json('data')));
    }

    public function testStore()
    {
        $this->asUser();

        $template = Template::factory()->create();
        $response = $this->postJson(route('content.pages.store'), [
            'title' => 'foo baz bar',
            'template_id' => $template->id,
        ]);

        $response->assertStatus(201);
        $this->assertTrue(Page::query()->where('title', 'foo baz bar')->exists());

        $response = $this->postJson(route('content.pages.store'), [
            'title' => 'lorem ipsum',
            'template_id' => null,
        ]);
        $response->assertStatus(201);
        $this->assertTrue(Page::query()->where('title', 'lorem ipsum')->exists());
    }

    public function testShow()
    {
        $this->asUser();

        $page = Page::factory()->create();
        $response = $this->getJson(route('content.pages.show', $page));

        $response->assertStatus(200);
        $this->assertEquals($page->id, $response->json('page.id'));
        $this->assertArrayHasKey('content', $response->json('page'));
        $this->assertArrayHasKey('versions', $response->json('page'));
    }

    public function testUpdateNoContent()
    {
        $this->asUser();

        $page = Page::factory()->create();
        $response = $this->putJson(route('content.pages.update', $page->id), [
            'title' => 'Lorem ipsum',
            'slug' => 'lorem-ipsum',
            'status' => PageStatus::PUBLISHED->value,
            'content' => [],
        ]);

        $response->assertStatus(200);
        $page->refresh();
        $this->assertEquals('Lorem ipsum', $page->title);
        $this->assertEquals('lorem-ipsum', $page->slug);
    }

    public function testUpdateHasContent()
    {
        $this->asUser();

        $global_content = GlobalContent::factory()->create();
        $element = ContentElement::factory()->create();
        $field = ContentElementField::factory()->create([
            'element_id' => $element->id,
            'slug' => 'lorem-ipsum',
        ]);
        $page = Page::factory()->create(['template_id' => null]);
        $response = $this->putJson(route('content.pages.update', $page->id), [
            'title' => 'Lorem ipsum ',
            'slug' => 'lorem-ipsum',
            'template_id' => null,
            'status' => PageStatus::PUBLISHED->value,
            'content' => [
                [
                    'id' => 1,
                    'slug' => 'lorem-ipsum',
                    'element_id' => $element->id,
                    'children' => [
                        [
                            'id' => 'abc123',
                            'slug' => 'lorem-ipsum',
                        ],
                    ]
                ],
                [
                    'id' => 'uuid-test',
                    'global_content_id' => $global_content->id,
                    'content' => 'ipsum',
                ],
            ],
        ]);

        $response->assertStatus(200);
        $page->refresh();
        $this->assertEquals('Lorem ipsum', $page->title);
        $this->assertEquals('lorem-ipsum', $page->slug);
    }

    public function testDestroy()
    {
        $this->asUser();

        $page = Page::factory()->create();
        $response = $this->deleteJson(route('content.pages.destroy', $page));

        $response->assertStatus(200);
        $this->assertFalse(Page::query()->where('id', $page->id)->exists());
    }

    public function testEmptyTrash()
    {
        $this->asUser();

        $pages = Page::factory(5)->create(['deleted_at' => Carbon::now()]);
        $response = $this->postJson(route('content.pages.empty-trash'), [
            'id' => [$pages->first()->id]
        ]);

        $response->assertStatus(200);
        $this->assertEquals(4, Page::query()->onlyTrashed()->count());

        $pages = Page::query()->onlyTrashed()->get();
        $response = $this->postJson(route('content.pages.empty-trash'), [
            'id' => $pages->pluck('id')
        ]);

        $response->assertStatus(200);
        $this->assertEquals(0, Page::query()->onlyTrashed()->count());
    }

    public function testCheckSlug()
    {
        $this->asUser();

        // good
        $page = Page::factory()->create();
        $response = $this->postJson(route('content.pages.check-slug', $page), [
            'slug' => 'foo-baz-bar',
        ]);
        $response->assertStatus(200);

        // bad slug
        $response = $this->postJson(route('content.pages.check-slug', $page), [
            'slug' => 'foo baz bar',
        ]);
        $response->assertStatus(422);
        $response = $this->postJson(route('content.pages.check-slug', $page), [
            'slug' => 'Foo-baz-bar',
        ]);
        $response->assertStatus(422);
        $response = $this->postJson(route('content.pages.check-slug', $page), [
            'slug' => 'foo_baz_bar',
        ]);
        $response->assertStatus(422);

        // Existing slug
        Page::factory()->create(['slug' => 'lorem-ipsum']);
        $response = $this->postJson(route('content.pages.check-slug', $page), [
            'slug' => 'lorem-ipsum',
        ]);
        $response->assertStatus(422);
    }

    public function testRestore()
    {
        $this->asUser();

        $page = Page::factory()->create(['deleted_at' => Carbon::now()]);
        $response = $this->postJson(route('content.pages.restore', $page));
        $response->assertStatus(200);
        $this->assertTrue($response->json('success'));
        $this->assertTrue(Page::query()->withoutTrashed()->where('id', $page->id)->exists());

        // Attempt to restore a page that doesn't exist
        $response = $this->postJson(route('content.pages.restore', 1123));
        $response->assertStatus(404);
        $this->assertFalse($response->json('success'));
    }

    public function testRestoreVersion()
    {
        $this->asUser();

        // Create a version and restore it, making sure the version is active after being attached to a page
        $page = Page::factory()->create();
        $version = Version::factory()->create([
            'page_id' => $page->id,
            'is_active' => false,
        ]);
        $response = $this->postJson(route('content.pages.restore-version', [$page, $version]));
        $response->assertStatus(200);
        $this->assertEquals($version->id, $response->json('data.id'));

        // Check the page version is now active
        $version->refresh();
        $this->assertTrue($version->is_active);

        // Attempt to restore a version that doesn't exist
        $response = $this->postJson(route('content.pages.restore-version', [$page, 132123]));
        $response->assertStatus(400);
        $this->assertFalse($response->json('success'));
    }
}
