<?php

namespace Mtc\ContentManager\Contracts;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\MorphMany;
use Illuminate\Support\Facades\Config;
use Mtc\ContentManager\Factories\GlobalContentFactory;
use Mtc\ContentManager\Models\Comment;
use Mtc\ContentManager\Models\Page;
use Mtc\ContentManager\Models\Template;
use Mtc\ContentManager\PageStatus;

abstract class GlobalContent extends Model
{
    use HasFactory;

    /**
     * Table name
     *
     * @var string
     */
    protected $table = 'global_content';

    /**
     * Mass assign attributes
     *
     * @var string[]
     */
    protected $fillable = [
        'language',
        'name',
        'content',
        'element_id',
    ];

    /**
     * Hide attributes from json
     *
     * @var string[]
     */
    protected $hidden = [
        'created_at',
        'updated_at',
    ];

    /**
     * Cast attributes to certain types
     *
     * @var string[]
     */
    protected $casts = [
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
    ];

    protected $appends = [
        'updated_at_formatted',
    ];

    /**
     * Model factory
     *
     * @return GlobalContentFactory
     */
    protected static function newFactory()
    {
        return GlobalContentFactory::new();
    }

    /**
     * Relationship with content on page where this is used
     *
     * @return HasMany
     */
    public function pageContent(): HasMany
    {
        return $this->hasMany(Config::get('pages.content_model'), 'global_content_id');
    }

    /**
     * Relationship with content element that describes this field
     *
     * @return BelongsTo
     */
    public function contentElement(): BelongsTo
    {
        return $this->belongsTo(Config::get('pages.element_model'), 'element_id');
    }

    /**
     * Relationship with users that are managing this content element
     *
     * @return BelongsToMany
     */
    public function users(): BelongsToMany
    {
        return $this->belongsToMany(Config::get('auth.providers.users.model'), 'global_content_users', 'content_id');
    }

    /**
     * Relationship with all comments that are attached to this global content
     *
     * @return MorphMany
     */
    public function comments(): MorphMany
    {
        return $this->morphMany(Config::get('pages.comment_model'), 'commentable');
    }

    /**
     * Relationship with open comments that are attached to this global content
     *
     * @return MorphMany
     */
    public function openComments(): MorphMany
    {
        return $this->morphMany(Config::get('pages.comment_model'), 'commentable')
            ->where('is_resolved', 0);
    }

    /**
     * Scope - withTemplateCount()
     * Adds template_usage column with count of all active templates using this global element
     *
     * @param Builder $query
     * @return Builder
     */
    public function scopeWithTemplateCount(Builder $query)
    {
        return $query->addSelect([
            'template_usage' => Template::query()
                ->selectRaw('COUNT(*)')
                ->whereHas(
                    'elements',
                    fn($query) => $query->whereColumn('global_element_id', $this->getTable() . '.id')
                )

        ]);
    }

    /**
     * Scope - withActiveUsage()
     * Adds active_pages column with all active page count using this global element
     *
     * @param Builder $query
     * @return Builder
     */
    public function scopeWithFreeformPageCount(Builder $query)
    {
        return $query->addSelect([
            'page_usage' => Page::query()
                ->selectRaw('COUNT(*)')
                ->where('status', PageStatus::PUBLISHED)
                ->whereHas('content', fn($query) => $query->whereColumn('global_element_id', $this->getTable() . '.id'))
        ]);
    }

    /**
     * Scope - withCommentCount()
     * Adds comment_count column with comment count for this element
     *
     * @param Builder $query
     * @return Builder
     */
    public function scopeWithCommentCount(Builder $query)
    {
        return $query->addSelect([
            'comment_count' => Comment::query()
                ->selectRaw('COUNT(*)')
                ->whereColumn('commentable_id', $this->getTable() . '.id')
                ->where('commentable_type', $this->getMorphClass())
        ]);
    }

    /**
     * Search term for query
     *
     * @param Builder $query
     * @param string $searchTerm
     * @return Builder
     */
    public function scopeSearchTerm(Builder $query, string $searchTerm): Builder
    {
        return $query->where('title', 'like', "%$searchTerm%")
            ->orWhere('subtitle', 'like', "%$searchTerm%");
    }

    /**
     * updated_at_formatted with diff from now
     *
     * @return string
     */
    public function getUpdatedAtFormattedAttribute()
    {
        return $this->updated_at->diffForHumans();
    }
}
