<?php

namespace Mtc\ContentManager;

use Carbon\Carbon;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;
use Illuminate\Support\Collection;
use Mtc\ContentManager\Contracts\Content;
use Mtc\ContentManager\Contracts\PageModel;
use Mtc\ContentManager\Contracts\TemplateElement;
use Mtc\ContentManager\Contracts\VersionModel;

class PageRepository
{
    /**
     * PageRepository constructor.
     * @param PageModel $page
     */
    public function __construct(protected PageModel $pageModel, protected VersionModel $version)
    {
        //
    }

    /**
     * @param string $title
     * @param int|null $templateId
     * @return PageModel|Model
     */
    public function create(string $title, int|null $templateId): PageModel
    {
        DB::beginTransaction();
        /** @var PageModel $page */
        $page = $this->pageModel->newQuery()
            ->create([
                'status' => 'draft',
                'title' => $title,
                'slug' => Str::slug($title),
                'template_id' => $templateId
            ]);

        if ($page->template_id) {
            $this->addContentFromLayout($page);
        }
        DB::commit();
        return $page;
    }

    /**
     * Save page data
     *
     * @param PageModel $page
     * @param array $input
     * @param int $author
     */
    public function save(PageModel $page, array $input, int $author)
    {
        $page->fill([
            'title' => $input['title'],
            'slug' => $input['slug'],
            'status' => $input['status'],
            'published_at' => $input['published_at'] ?? null,
            'seo' => $input['seo'] ?? [],
            'meta' => $input['meta'] ?? [],
        ])->save();

        $this->saveContent($page, $input['content']);
        $this->savePageVersion($page, $input['content'], $author);
    }

    /**
     * Mark page as removed
     *
     * @param int|array $id
     * @param bool $force
     * @return mixed
     */
    public function remove(int|array $id, bool $force = false)
    {
        return $this->pageModel->newQuery()
            ->whereIn('id', (array)$id)
            ->when(
                $force,
                fn($query) => $query->forceDelete(),
                fn($query) => $query->delete()
            );
    }

    /**
     * Update published state of a page
     *
     * @param int|array $id
     * @param Carbon|null $value
     * @return int
     */
    public function setPublishState(int|array $id, Carbon|null $value)
    {
        return $this->pageModel->newQuery()
            ->whereIn('id', (array)$id)
            ->update([
                'status' => $value ? PageStatus::PUBLISHED->value : PageStatus::DRAFT->value,
                'published_at' => $value
            ]);
    }

    /**
     * Restore a deleted page
     *
     * @param int $pageId
     * @return bool
     */
    public function restorePage(int $pageId): bool
    {
        return $this->pageModel
            ->newQuery()
            ->onlyTrashed()
            ->findOrFail($pageId)
            ->restore();
    }

    /**
     * Restore content from version history
     *
     * @param int $version_id
     */
    public function restoreVersion(int $version_id)
    {
        /** @var VersionModel $version */
        $version = $this->version->newQuery()->findOrFail($version_id);
        $this->saveContent($version->page, $version->content ?? []);
        $version->is_active = true;
        $version->save();
    }

    /**
     * Create a copy page
     *
     * @param int $pageId
     * @param string $title
     * @param bool $withContent
     * @return PageModel
     */
    public function copyPage(int $pageId, string $title, bool $withContent)
    {
        /** @var PageModel $original */
        $original = $this->pageModel->newQuery()->findOrFail($pageId);
        DB::beginTransaction();
        /** @var PageModel $page */
        $page = $this->pageModel->newQuery()
            ->create([
                'status' => 'draft',
                'title' => $title,
                'slug' => Str::slug($title),
                'template_id' => $original->template_id
            ]);

        if ($withContent) {
            $original->content->each(fn (Content $content) => $page->content()->create($content->toArray()));
        } elseif ($page->template_id) {
            $this->addContentFromLayout($page);
        }
        DB::commit();
        return $page;
    }

    /**
     * Save a new version of the page
     *
     * @param PageModel $page
     * @param array $content
     * @param int $author
     * @param false $is_active
     */
    protected function savePageVersion(PageModel $page, array $content, int $author, bool $is_active = false): void
    {
        if ($this->hasPageContentChanged($page, $content) === false) {
            return;
        }

        $page->versions()
            ->create([
                'author_id' => $author,
                'is_active' => $is_active,
                'content' => $content
            ]);
    }

    /**
     * Persist page content to database
     *
     * @param PageModel $page
     * @param array $content
     */
    protected function saveContent(PageModel $page, array $content)
    {
        collect($content)
            ->each(fn($element) => $this->saveContentElement($page, $element));
    }

    /**
     * Save content element against page or its content element
     *
     * @param PageModel $page
     * @param array $element
     */
    protected function saveContentElement(PageModel|Content $parent, array $element)
    {
        /** @var Content $entry */
        $entry = $parent instanceof PageModel
            ? $parent->content()->updateOrCreate(['id' => $element['id'] ?? null], $element)
            : $parent->subContent()->updateOrCreate(['id' => $element['id'] ?? null], $element);

        $this->syncContentElementMedia(collect($element['media'] ?? []), $entry);
        collect($element['subContent'] ?? [])->each(fn($content) => $this->saveContentElement($entry, $content));
    }

    /**
     * Set media elements on Content
     *
     * @param Collection $element
     * @param Content $entry
     * @return void
     */
    protected function syncContentElementMedia(Collection $media_entries, Content $entry)
    {
        $entry->mediaUses()
            ->whereNotIn('id', $media_entries->pluck('id'))
            ->delete();

        $media_entries->each(fn($media) => $entry->mediaUses()->updateOrCreate(['id' => $media['id']], $media));
    }

    /**
     * Check if content on page has changed requiring to save a new version
     *
     * @param PageModel $page
     * @param array $current_content
     * @param int|null $version_id
     * @return bool
     */
    protected function hasPageContentChanged(PageModel $page, array $current_content, int $version_id = null): bool
    {
        $version = $page->versions()
            ->when(
                $version_id,
                fn($query) => $query->where('id', $version_id),
                fn($query) => $page->where('is_active', 1)
            )
            ->first();


        return ($version->content ?? []) !== $current_content;
    }

    /**
     * Create content fields for page based on layout
     *
     * @param PageModel $page
     * @return void
     */
    protected function addContentFromLayout(PageModel $page)
    {
        $page->template->elements()
            ->get()
            ->each(fn(TemplateElement $element) => $page->content()
                ->updateOrCreate(
                    [ 'template_element_id' => $element->id ],
                    $this->templateElementToContent($element)
                ));
    }

    /**
     * Convert Template element to Page content object
     *
     * @param TemplateElement $element
     * @return array
     */
    protected function templateElementToContent(TemplateElement $element): array
    {
        $data = $element->toArray();
        $data['template_element_id'] = $element->id;
        unset($data['id']);
        return $data;
    }
}
