<?php

namespace Mtc\ContentManager\Http\Controllers;

use Illuminate\Foundation\Validation\ValidatesRequests;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Config;
use Mtc\ContentManager\Contracts\Menu;
use Mtc\ContentManager\Http\Requests\StoreMenuRequest;
use Mtc\ContentManager\Http\Requests\UpdateMenuRequest;
use Mtc\ContentManager\Http\Resources\MenuList;
use Mtc\ContentManager\Http\Resources\MenuView;
use Mtc\ContentManager\Facades\Menu as MenuFacade;

class MenuController
{
    use ValidatesRequests;

    /**
     * List menus
     *
     * @param menu $menu
     * @return MenuList
     */
    public function index(Request $request, Menu $menu): MenuList
    {
        $resource = Config::get('pages.menu_list_resource');
        return new $resource($menu->newQuery()->setSortBy($request->input('sort_by'))->paginate());
    }

    /**
     * Store a new menu
     *
     * @param StoreMenuRequest $request
     * @param Menu $menu
     * @return MenuView
     */
    public function store(StoreMenuRequest $request, Menu $menu): MenuView
    {
        $record = $menu->newQuery()->create($request->validated());
        $resource = Config::get('pages.menu_view_resource');
        return new $resource($record->refresh());
    }

    /**
     * Display menu record
     *
     * @param Menu $menu
     * @return MenuView
     */
    public function show(Menu $menu): MenuView
    {
        $resource = Config::get('pages.menu_view_resource');
        return new $resource($menu);
    }

    /**
     * Update the specified menu
     *
     * @param \Illuminate\Http\Request $request
     * @param Menu $menu
     * @return MenuView
     */
    public function update(UpdateMenuRequest $request, Menu $menu): MenuView
    {
        $resource = Config::get('pages.menu_view_resource');
        return new $resource(MenuFacade::update($menu, $request));
    }

    /**
     * Remove the menu
     *
     * @param Menu $menu
     * @return bool
     */
    public function destroy(Menu $menu): bool
    {
        return $menu->delete();
    }
}
