<?php

namespace Mtc\ContentManager\Contracts;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Support\Facades\Config;
use Mtc\ContentManager\Factories\ContentElementFactory;
use Mtc\ContentManager\Traits\ModelSortAndFilter;

abstract class ContentElement extends Model
{
    use HasFactory;
    use ModelSortAndFilter;

    /**
     * Table name
     *
     * @var string
     */
    protected $table = 'content_elements';

    /**
     * Mass assign attributes
     *
     * @var string[]
     */
    protected $fillable = [
        'title',
        'subtitle',
        'icon',
        'category',
        'is_enabled',
    ];

    protected $casts = [
        'is_enabled' => 'bool',
    ];

    /**
     * Hide attributes from json
     *
     * @var string[]
     */
    protected $hidden = [
        'created_at',
        'updated_at',
    ];

    /**
     * Model factory
     *
     * @return ContentElementFactory
     */
    protected static function newFactory()
    {
        return ContentElementFactory::new();
    }

    /**
     * Templates that use this content element
     *
     * @return HasMany
     */
    public function templateElements(): HasMany
    {
        return $this->hasMany(Config::get('pages.template_element_model'), 'element_id');
    }

    /**
     * Templates that use this content element
     *
     * @return HasMany
     */
    public function globalElements(): HasMany
    {
        return $this->hasMany(Config::get('pages.global_content_model'), 'element_id');
    }

    /**
     * Templates that use this content element
     *
     * @return HasMany
     */
    public function pageElements(): HasMany
    {
        return $this->hasMany(Config::get('pages.global_content_model'), 'element_id');
    }

    /**
     * Fields that make this element
     *
     * @return HasMany
     */
    public function fields(): HasMany
    {
        return $this->hasMany(Config::get('pages.element_field_model'), 'element_id');
    }

    /**
     * Search term for query
     *
     * @param Builder $query
     * @param string $searchTerm
     * @return Builder
     */
    public function scopeSearchTerm(Builder $query, string $searchTerm): Builder
    {
        return $query->where('title', 'like', "%$searchTerm%")
            ->orWhere('subtitle', 'like', "%$searchTerm%");
    }
}
