<?php

namespace Mtc\ContentManager\Http\Controllers;

use Illuminate\Foundation\Validation\ValidatesRequests;
use Illuminate\Http\Request;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Mtc\ContentManager\Contracts\ContentElement;
use Mtc\ContentManager\ElementRepository;
use Mtc\ContentManager\Http\Requests\CopyContentElementRequest;
use Mtc\ContentManager\Http\Requests\StoreContentElementRequest;
use Mtc\ContentManager\Http\Requests\UpdateContentElementRequest;
use Mtc\ContentManager\Http\Resources\ContentElementList;
use Mtc\ContentManager\Http\Resources\ContentElementView;

class ElementController
{
    use ValidatesRequests;

    /**
     * Display a listing of content elements in system.
     *
     * @param Request $request
     * @param ElementRepository $repository
     * @return ContentElementList
     */
    public function index(Request $request, ElementRepository $repository): ContentElementList
    {
        $resource = Config::get('pages.element_list_resource');
        return new $resource($repository->getList($request->input('search-term', ''), $request->input('sort_by', '')));
    }

    /**
     * Show the content element in detail
     *
     * @param int $elementId
     * @param ElementRepository $repository
     * @return ContentElementView
     */
    public function show(int $elementId, ElementRepository $repository): ContentElementView
    {
        $resource = Config::get('pages.element_view_resource');
        return new $resource($repository->find($elementId));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param StoreContentElementRequest $request
     * @param ElementRepository $repository
     * @return ContentElementView
     */
    public function store(StoreContentElementRequest $request, ElementRepository $repository): ContentElementView
    {
        $element = $repository->create(
            title: $request->input('title'),
        );
        $resource = Config::get('pages.element_view_resource');
        return new $resource($element);
    }

    /**
     * Update content element
     *
     * @param UpdateContentElementRequest $request
     * @param ElementRepository $repository
     * @return ContentElementView
     */
    public function update(
        UpdateContentElementRequest $request,
        ContentElement $element,
        ElementRepository $repository
    ): ContentElementView {

        if (config('pages.use_transactions')) {
            DB::beginTransaction();
        }
        $element = $repository->update(
            element: $element,
            title: $request->input('title'),
            subtitle: $request->input('subtitle'),
            icon: $request->input('icon'),
            category: $request->input('category'),
            enabled: $request->input('is_enabled', 0),
            fields: $request->input('fields', [])
        );

        if ($request->hasFile('icon_file')) {
            Storage::disk('content-icons')
                ->put($request->file('icon_file')->getClientOriginalName(), $request->file('icon_file'));
        }

        if (config('pages.use_transactions')) {
            DB::commit();
        }

        $resource = Config::get('pages.element_view_resource');
        return new $resource($element);
    }

    /**
     * Create a new copy of an existing content element
     *
     * @param CopyContentElementRequest $request
     * @param int $id_to_copy
     * @param ElementRepository $repository
     * @return ContentElementView
     */
    public function copy(
        CopyContentElementRequest $request,
        int $id_to_copy,
        ElementRepository $repository
    ): ContentElementView {
        $element = $repository->makeCopy($id_to_copy, $request->input('title'));
        $resource = Config::get('pages.element_view_resource');
        return new $resource($element);
    }

    /**
     * Remove the content element.
     *
     * @param int $elementId
     * @param ElementRepository $repository
     * @return array
     */
    public function destroy(int $elementId, ElementRepository $repository): array
    {
        return [
            'status' => $repository->destroy($elementId) ? 'success' : 'failed',
        ];
    }

    /**
     * Retrieve element usage - check element before saving to prevent breaking changes.
     * Usage limits 10 records per type
     *
     * @param int $elementId
     * @param ElementRepository $repository
     * @return int[]
     */
    public function checkUsage(int $elementId, ElementRepository $repository)
    {
        return $repository->getUsage($elementId);
    }

    /**
     * Show full usage of a content element for the content type
     *
     * @param int $elementId
     * @param string $type
     * @param ElementRepository $repository
     * @return Collection
     */
    public function fullUsageList(int $elementId, string $type, ElementRepository $repository): Collection
    {
        return $repository->fullUsageList($elementId, $type);
    }

    /**
     * Show list of icons available for content elements
     *
     * @return array
     */
    public function icons(): array
    {
        return Storage::disk('content-icons')->allFiles();
    }
}
