<?php

namespace Mtc\ContentManager\Contracts;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Support\Facades\Config;
use Mtc\ContentManager\Factories\TemplateElementFactory;

abstract class TemplateElement extends Model
{
    use HasFactory;

    /**
     * Table name
     *
     * @var string
     */
    protected $table = 'page_template_elements';

    /**
     * Mass assign attributes
     *
     * @var string[]
     */
    protected $fillable = [
        'template_id',
        'global_content_id',
        'element_id',
        'name',
        'description',
        'hint',
        'is_list',
        'max_repeat_count',
        'order',
    ];

    /**
     * Hide attributes from json
     *
     * @var string[]
     */
    protected $hidden = [
        'created_at',
        'updated_at',
    ];

    /**
     * Cast attributes to certain types
     *
     * @var string[]
     */
    protected $casts = [
        'is_list' => 'bool',
    ];

    /**
     * Model factory
     *
     * @return TemplateElementFactory
     */
    protected static function newFactory()
    {
        return TemplateElementFactory::new();
    }

    /**
     * Relationship with template that has this element
     *
     * @return BelongsTo
     */
    public function template(): BelongsTo
    {
        return $this->belongsTo(Config::get('pages.template_model'), 'template_id');
    }

    /**
     * Relationship with content elements that use this page template content element
     *
     * @return BelongsTo
     */
    public function element(): BelongsTo
    {
        return $this->belongsTo(Config::get('pages.content_element_model'), 'element_id');
    }

    /**
     * Relationship with page content elements that are built using this template element
     *
     * @return HasMany
     */
    public function contentElements(): HasMany
    {
        return $this->hasMany(Config::get('pages.content_model'), 'template_element_id');
    }

    /**
     * Relationship with global content element
     *
     * @return BelongsTo
     */
    public function globalContent(): BelongsTo
    {
        return $this->belongsTo(Config::get('pages.global_content_model'), 'global_content_id');
    }
}
