<?php

namespace Mtc\ContentManager\Tests\Feature;

use Illuminate\Foundation\Testing\RefreshDatabase;
use Mtc\ContentManager\Models\ContentElement;
use Mtc\ContentManager\Models\Template;
use Mtc\ContentManager\Tests\AsUser;
use Mtc\ContentManager\Tests\TestCase;

class TemplateControllerTest extends TestCase
{
    use RefreshDatabase;
    use AsUser;

    public function testIndex()
    {
        $this->asUser();

        Template::factory(5)->create();
        $response = $this->getJson(route('content.templates.index'));

        $response->assertStatus(200);
        $this->assertEquals(5, count($response->json('data')));
    }

    public function testStore()
    {
        $this->asUser();

        $response = $this->postJson(route('content.templates.store'), [
            'name' => 'lorem ipsum'
        ]);

        $response->assertStatus(201);
        $this->assertTrue(Template::query()->where('name', 'lorem ipsum')->exists());
    }

    public function testShow()
    {
        $this->asUser();

        $template = Template::factory()->create();
        $response = $this->getJson(route('content.templates.show', $template));

        $response->assertStatus(200);
        $this->assertArrayHasKey('template', $response->json());
        $this->assertArrayHasKey('template_elements', $response->json());
        $this->assertArrayHasKey('elements', $response->json());
        $this->assertArrayHasKey('comments', $response->json());
    }

    public function testUpdate()
    {
        $this->asUser();

        $element = ContentElement::factory()->create();
        $template = Template::factory()->create();
        $response = $this->putJson(route('content.templates.update', $template), [
            'name' => 'foo',
            'description' => 'lorem ipsum',
            'elements' => [
                [
                    'element_id' => $element->id,
                    'global_content_id' => null,
                    'name' => 'foo',
                ],
            ],
        ]);

        $response->assertStatus(200);
        $template->refresh();
        $this->assertEquals('foo', $template->name);
        $this->assertEquals('lorem ipsum', $template->description);
        $this->assertEquals(1, $template->elements()->count());
    }

    public function testDestroy()
    {
        $this->asUser();

        $template = Template::factory()->create();
        $response = $this->deleteJson(route('content.templates.destroy', $template));

        $response->assertStatus(200);
        $this->assertFalse(Template::query()->where('id', $template->id)->exists());
    }

    public function testUnused()
    {
        $this->asUser();

        $template = Template::factory()->create();
        $response = $this->getJson(route('content.templates.unused'));

        $response->assertStatus(200);
        $this->assertEquals(1, count($response->json('data')));
        $this->assertEquals($template->id, $response->json('data.0.id'));
    }

    public function testCopy()
    {
        $this->asUser();

        $element1 = ContentElement::factory()->create();
        $element2 = ContentElement::factory()->create();
        $element3 = ContentElement::factory()->create();
        $template = Template::factory()->create();
        $template->elements()
            ->create([
                'element_id' => $element1->id,
                'name' => 'foo',
            ]);
        $template->elements()
            ->create([
                'element_id' => $element2->id,
                'name' => 'baz',
            ]);
        $template->elements()
            ->create([
                'element_id' => $element3->id,
                'name' => 'bar',
            ]);

        $response = $this->postJson(route('content.templates.copy', $template), [
            'name' => 'baz',
        ]);

        $response->assertStatus(201);
        $copy = Template::query()->findOrFail($response->json('id'));
        $this->assertEquals(3, $copy->elements()->count());
    }
}
