<?php

namespace Mtc\ContentManager\Http\Controllers;

use Illuminate\Contracts\Pagination\LengthAwarePaginator;
use Illuminate\Foundation\Validation\ValidatesRequests;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Mtc\ContentManager\Contracts\GlobalContent;
use Mtc\ContentManager\Http\Requests\CopyGlobalContentElementRequest;
use Mtc\ContentManager\Http\Requests\RemoveGlobalContentRequest;
use Mtc\ContentManager\Http\Requests\StoreGlobalContentRequest;
use Mtc\ContentManager\Http\Requests\UpdateGlobalContentRequest;
use Mtc\ContentManager\Http\Resources\GlobalContentElementView;

class GlobalElementController
{
    use ValidatesRequests;

    /**
     * Show list of global elements
     *
     * @param Request $request
     * @param GlobalContent $model
     * @return LengthAwarePaginator
     */
    public function index(Request $request, GlobalContent $model)
    {
        return $model->newQuery()
            ->when($request->filled('search-term'), fn($query) => $query->searchTerm($request->input('search-term')))
            ->with('users')
            ->withTemplateCount()
            ->withFreeformPageCount()
            ->withCommentCount()
            ->paginate();
    }

    /**
     * Show a global content element
     *
     * @param GlobalContent $content_block
     * @param int $block_id
     * @return \Illuminate\Http\Response|array
     */
    public function show(GlobalContent $content_block, int $block_id): GlobalContentElementView
    {
        $block = $content_block->newQuery()
            ->with([
                'contentElement.fields',
                'openComments.user',
            ])
            ->find($block_id);

        return new GlobalContentElementView($block);
    }

    /**
     * Store a newly created global content element.
     *
     * @param StoreGlobalContentRequest $request
     * @return GlobalContentElementView
     */
    public function store(StoreGlobalContentRequest $request, GlobalContent $globalContent): GlobalContentElementView
    {
        $element = $globalContent->newQuery()
            ->findOrNew($request->input('id'))
            ?->fill($request->input());

        $element->save();

        return new GlobalContentElementView($element);
    }

    /**
     * Update a global content element
     *
     * @param UpdateGlobalContentRequest $request
     * @param GlobalContent $globalContent
     */
    public function update(UpdateGlobalContentRequest $request, GlobalContent $globalContent, int $elementId)
    {
        DB::beginTransaction();
        $element = $globalContent->newQuery()
            ->findOrFail($elementId)
            ?->fill($request->input());

        $element->save();
        DB::commit();
        return new GlobalContentElementView($element);
    }

    /**
     * Remove the specified global content element
     *
     * @param int $id
     * @return void
     */
    public function destroy(RemoveGlobalContentRequest $request, GlobalContent $model, int $id)
    {
        $block = $model->newQuery()->find($id)?->delete();
    }

    /**
     * Create a new copy of an existing global content element
     *
     * @param CopyGlobalContentElementRequest $request
     * @param int $id_to_copy
     * @return GlobalContentElementView
     */
    public function copy(CopyGlobalContentElementRequest $request, GlobalContent $element): GlobalContentElementView
    {
        $copy = $element->newQuery()->create([
            'name' => $request->input('name'),
            'element_id' => $element->element_id,
            'content' => $request->input('copy_content') ? $element->content : null,
        ]);
        return new GlobalContentElementView($copy);
    }
}
