<?php

namespace Mtc\ContentManager\Tests\Unit;

use Illuminate\Foundation\Testing\RefreshDatabase;
use Mtc\ContentManager\ElementRepository;
use Mtc\ContentManager\Models\ContentElement;
use Mtc\ContentManager\Models\ContentElementField;
use Mtc\ContentManager\Models\GlobalContent;
use Mtc\ContentManager\Models\Page;
use Mtc\ContentManager\Models\Template;
use Mtc\ContentManager\Tests\TestCase;

class ElementRepositoryTest extends TestCase
{
    use RefreshDatabase;

    public function testGetUsage(): void
    {
        $repository = new ElementRepository(new ContentElement());
        $element = $repository->create('Text editor', 'WYSIWYG editor', 'fa fa-letters', 'Content', true);

        $usage = $repository->getUsage($element->id);

        $this->assertArrayHasKey('templates', $usage);
        $this->assertArrayHasKey('freeform_pages', $usage);
        $this->assertArrayHasKey('global_elements', $usage);

        // add actual use cases for query verification

        $this->assertEquals(0, $usage['templates']->count());
        $this->assertEquals(0, $usage['freeform_pages']->count());
        $this->assertEquals(0, $usage['global_elements']->count());

        $page = Page::factory()->create([
            'title' => 'hello world',
            'template_id' => null
        ]);

        $page->content()->create([
            'element_id' => $element->id,
            'name' => 'foo',
        ]);

        $usage = $repository->getUsage($element->id);
        $this->assertEquals(1, $usage['freeform_pages']->count());

        $template = Template::factory()->create([]);
        $template->elements()->create([
            'element_id' => $element->id,
            'name' => 'foo',
        ]);
        $usage = $repository->getUsage($element->id);
        $this->assertEquals(1, $usage['templates']->count());


        GlobalContent::factory()
            ->create([
                'element_id' => $element->id
            ]);

        $usage = $repository->getUsage($element->id);
        $this->assertEquals(1, $usage['global_elements']->count());
    }

    public function testCreateEmptyElement()
    {
        $repository = new ElementRepository(new ContentElement());
        $element = $repository->create('Text editor', 'WYSIWYG editor', 'fa fa-letters', 'Content', true);

        $exists = ContentElement::query()
            ->where('title', 'Text editor')
            ->where('subtitle', 'WYSIWYG editor')
            ->where('icon', 'fa fa-letters')
            ->where('category', 'Content')
            ->exists();

        $this->assertTrue($exists);
    }

    public function testAddSimpleTextElement()
    {
        $repository = new ElementRepository(new ContentElement());

        $element = $repository->create('Text editor', 'WYSIWYG editor', 'fa fa-letters', 'Content', true);

        $repository->setModel($element)
            ->addField([
                'name' => 'Text field',
                'type' => 'editor'
            ]);

        $exists = ContentElementField::query()
            ->where('field_type', 'editor')
            ->where('name', 'Text field')
            ->where('element_id', $element->id)
            ->exists();

        $this->assertTrue($exists);
    }

    public function testAddElementWithMultipleFields(): void
    {
        $repository = new ElementRepository(new ContentElement());

        $element = $repository->create('Link', 'Simple Link', '', 'Content', true);

        $repository->setModel($element)
            ->addField([
                'name' => 'Link Text',
                'type' => 'text'
            ])
            ->addField([
                'name' => 'Link anchor',
                'type' => 'url'
            ]);

        $this->assertEquals(2, $element->fields()->count());
    }
}
