<?php

namespace Mtc\ContentManager\Contracts;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\MorphTo;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Mtc\ContentManager\Facades\Media as MediaFacade;
use Mtc\ContentManager\Factories\MediaUseFactory;
use Mtc\ContentManager\ImageSize;
use Mtc\ContentManager\Models\MediaSize;

/**
 * @property ?Media $media
 * @property string $owner_type
 * @property string $model
 */
class MediaUse extends Model
{
    use HasFactory;

    /**
     * Table name
     *
     * @var string
     */
    protected $table = 'media_uses';

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'media_id',
        'owner_type',
        'owner_id',
        'alt_text',
        'title',
        'description',
        'caption',
        'dimensions',
        'allowed_sizes',
        'data',
        'order',
        'primary',
        'secondary',
        'flags',
    ];

    /**
     * Hide attributes from json
     *
     * @var string[]
     */
    protected $hidden = [
        'created_at',
        'updated_at'
    ];

    /**
     * Cast attributes to certain types
     *
     * @var string[]
     */
    protected $casts = [
        'allowed_sizes' => 'array',
        'flags' => 'array',
    ];

    /**
     * Model Factory
     * @return mixed
     */
    protected static function newFactory()
    {
        return MediaUseFactory::new();
    }

    /**
     * Relationship with the media file
     *
     * @return BelongsTo
     */
    public function media(): BelongsTo
    {
        return $this->belongsTo(Config::get('media.media_model'), 'media_id');
    }

    /**
     * Relationship with the owner of the media file
     *
     * @return MorphTo
     */
    public function owner(): MorphTo
    {
        return $this->morphTo('owner');
    }

    /**
     * Get the url for media use for given dimensions
     *
     * @param string $dimensions
     * @return string|null
     */
    public function getUrl(string $dimensions): ?string
    {
        if ($this->media->external) {
            return $this->media->externalUrl($dimensions);
        }

        $dimension_array = MediaFacade::pathIsWidthAndHeight($dimensions)
            ? MediaFacade::pathStringToArray($dimensions)
            : $this->nameToDimensions($dimensions);

        if ($dimension_array) {
            $size = ImageSize::fromArray($dimension_array);
            return Storage::disk('media')->url($size->pathOnDisk($this->media->src));
        }

        return null;
    }

    private function nameToDimensions(string $dimensions): ?array
    {
        if (array_key_exists($dimensions, \config('media.thumbnail_sizes'))) {
            list($width, $height) = explode('x', \config('media.thumbnail_sizes.' . $dimensions));
            return compact('width', 'height');
        }

        return Cache::rememberForever(
            "media-size-{$this->owner_type}-$dimensions",
            fn() => MediaSize::query()
                ->where('model', $this->owner_type)
                ->where('label', $dimensions)
                ->first()
                ?->toArray()
        );
    }
}
