<?php

namespace Mtc\ContentManager\Http\Controllers;

use Illuminate\Foundation\Validation\ValidatesRequests;
use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Illuminate\Support\Facades\Config;
use Mtc\ContentManager\Contracts\Menu;
use Mtc\ContentManager\Http\Requests\CopyMenuRequest;
use Mtc\ContentManager\Http\Requests\StoreMenuRequest;
use Mtc\ContentManager\Http\Requests\UpdateMenuRequest;
use Mtc\ContentManager\Http\Resources\MenuList;
use Mtc\ContentManager\Http\Resources\MenuView;
use Mtc\ContentManager\Facades\Menu as MenuFacade;

class MenuController
{
    use ValidatesRequests;

    /**
     * List menus
     *
     * @param menu $menu
     * @return MenuList
     */
    public function index(Request $request, Menu $menu): MenuList
    {
        $resource = Config::get('pages.menu_list_resource');
        return new $resource($menu->newQuery()
            ->setSortBy($request->input('sort_by'))
            ->setFilters($request->input('filters') ?? '')
            ->paginate());
    }

    /**
     * Store a new menu
     *
     * @param StoreMenuRequest $request
     * @param Menu $menu
     * @return MenuView
     */
    public function store(StoreMenuRequest $request, Menu $menu): MenuView
    {
        $record = MenuFacade::create($menu, $request);
        $resource = Config::get('pages.menu_view_resource');
        return new $resource($record->refresh());
    }

    /**
     * Display menu record
     *
     * @param Menu $menu
     * @return MenuView
     */
    public function show(Menu $menu): MenuView
    {
        $resource = Config::get('pages.menu_view_resource');
        return new $resource($menu);
    }

    /**
     * Update the specified menu
     *
     * @param \Illuminate\Http\Request $request
     * @param Menu $menu
     * @return MenuView
     */
    public function update(UpdateMenuRequest $request, Menu $menu): MenuView
    {
        $resource = Config::get('pages.menu_view_resource');
        return new $resource(MenuFacade::update($menu, $request));
    }

    /**
     * Remove the menu
     *
     * @param Menu $menu
     * @return Response
     */
    public function destroy(Menu $menu): Response
    {
        if (MenuFacade::canRemove($menu)) {
            return response([
                'success' => $menu->delete(),
            ]);
        }

        return response([
            'message' => __('validation.not_allowed_to_delete')
        ], 422);
    }

    public function copy(CopyMenuRequest $request, Menu $menu)
    {
        $resource = Config::get('pages.menu_view_resource');
        return new $resource(MenuFacade::copy($menu, $request));
    }
}
