<?php

namespace Mtc\ContentManager\Tests\Feature;

use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Http\JsonResponse;
use Mtc\ContentManager\Models\ContentElement;
use Mtc\ContentManager\Models\Menu;
use Mtc\ContentManager\Tests\AsUser;
use Mtc\ContentManager\Tests\TestCase;
use Mtc\ContentManager\Tests\User;

class MenuControllerTest extends TestCase
{
    use RefreshDatabase;
    use AsUser;

    public function testListMenus()
    {
        $this->asUser();
        /** @var JsonResponse $response */
        $response = $this->get('/api/content/menus');

        $response->assertStatus(200);
        $body = $response->getData(true);
        $this->assertArrayHasKey('current_page', $body);
        $this->assertArrayHasKey('data', $body);
        $this->assertEquals(0, count($body['data']));

        Menu::factory()->create(['title' => 'foo']);
        Menu::factory()->create(['title' => 'baz']);
        Menu::factory()->create(['title' => 'bar']);

        $response = $this->get('/api/content/menus');

        $body = $response->getData(true);
        $this->assertEquals(3, count($body['data']));
    }

    public function testAddMenu()
    {
        $this->asUser();
        /** @var JsonResponse $response */
        $response = $this->post('/api/content/menus', [
            'title' => 'Lorem ipsum'
        ]);

        $response->assertStatus(201);
        $body = $response->getData(true);
        $this->assertEquals('Lorem ipsum', $body['title']);
    }

    public function testShowMenu()
    {
        $this->asUser();

        $menu = Menu::factory()->create(['title' => 'foo baz']);

        /** @var JsonResponse $response */
        $response = $this->get('/api/content/menus/' . $menu->id);

        $response->assertStatus(200);
        $body = $response->getData(true);
        $this->assertEquals(1, $body['id']);
        $this->assertEquals('foo baz', $body['title']);
    }

    public function testUpdateEmptyMenu()
    {
        $this->asUser();

        $menu = Menu::factory()->create(['title' => 'foo baz']);

        /** @var JsonResponse $response */
        $element = ContentElement::factory()->create();
        $response = $this->putJson(route('content.menus.update', $menu->id), [
            'title' => 'lorem ipsum',
            'slug' => 'lorem-ipsum',
            'first_child_element_id' => $element->id,
            'is_enabled' => true,
            'entries' => [],
        ]);
        $response->assertStatus(200);
        $menu->refresh();
        $this->assertEquals('lorem ipsum', $menu->title);
        $this->assertEquals($element->id, $menu->first_child_element_id);
        $this->assertEquals(true, $menu->is_enabled);
    }


    public function testUpdateMenuWithEntries()
    {
        $this->asUser();

        $menu = Menu::factory()->create(['title' => 'foo baz']);

        /** @var JsonResponse $response */
        $element = ContentElement::factory()->create();
        $response = $this->putJson(route('content.menus.update', $menu->id), [
            'title' => 'lorem ipsum',
            'slug' => 'lorem-ipsum',
            'first_child_element_id' => $element->id,
            'is_enabled' => true,
            'entries' => [
                [
                    'is_enabled' => true,
                    'name' => 'foo',
                    'url' => '/baz',
                    'content' => [],
                    'data' => [],
                ]
            ],
        ]);

        $response->assertStatus(200);

        $menu->refresh();
        $this->assertEquals(1, $menu->entries()->count());
    }

    public function testRemoveMenu()
    {
        $this->asUser();
        $menu = Menu::factory()->create();

        $response = $this->get('/api/content/menus/' . $menu->id);
        $response->assertStatus(200);

        /** @var JsonResponse $response */
        $response = $this->delete('/api/content/menus/' . $menu->id);
        $response->assertStatus(200);

        $response = $this->get('/api/content/menus/' . $menu->id);
        $response->assertStatus(404);
    }
}
