<?php

namespace Mtc\ContentManager\Tests\Feature;

use Illuminate\Foundation\Testing\RefreshDatabase;
use Mtc\ContentManager\Contracts\ContentElementField;
use Mtc\ContentManager\Models\Content;
use Mtc\ContentManager\Models\ContentElement;
use Mtc\ContentManager\Models\GlobalContent;
use Mtc\ContentManager\Models\Page;
use Mtc\ContentManager\Models\Template;
use Mtc\ContentManager\Models\TemplateElement;
use Mtc\ContentManager\Models\Version;
use Mtc\ContentManager\PageStatus;
use Mtc\ContentManager\Tests\AsUser;
use Mtc\ContentManager\Tests\TestCase;

class PageContentTest extends TestCase
{
    use RefreshDatabase;
    use AsUser;

    public function testWorkflow()
    {
        $this->asUser();
        $slider = $this->postJson(route('content.elements.store'), ['title' => 'Slider']);
        $slider->assertStatus(201);

        $response = $this->putJson(route('content.elements.update', $slider->json('data.id')), [
            'title' => 'Slider',
            'subtitle' => 'baz',
            'category' => 'bar',
            'icon' => '/path/to/icon',
            'is_enabled' => true,
            'fields' => [
                [
                    'id' => 'test-1',
                    'name' => 'Image',
                    'field_type' => 'image'
                ],
                [
                    'id' => 'test-2',
                    'name' => 'Title',
                    'field_type' => 'text'
                ],
                [
                    'id' => 'test-3',
                    'name' => 'Link',
                    'field_type' => 'text'
                ],
            ]
        ]);
        $response->assertStatus(200);

        $text = $this->postJson(route('content.elements.store'), ['title' => 'Text Block']);
        $text->assertStatus(201);

        $response = $this->putJson(route('content.elements.update', $text->json('data.id')), [
            'title' => 'Text Block',
            'subtitle' => 'baz',
            'category' => 'bar',
            'icon' => '/path/to/icon',
            'is_enabled' => true,
            'fields' => [
                [
                    'id' => 'test-1',
                    'name' => 'Title',
                    'field_type' => 'text'
                ],
                [
                    'id' => 'test-2',
                    'name' => 'Text',
                    'field_type' => 'textarea'
                ],
            ]
        ]);
        $response->assertStatus(200);

        $global = $this->postJson(route('content.global-elements.store'), [
            'name' => 'lorem ipsum',
            'element_id' => $text->json('data.id'),
        ]);
        $global->assertStatus(201);

        $template = $this->postJson(route('content.templates.store'), ['name' => 'Default Template']);
        $template->assertStatus(201);

        $response = $this->putJson(route('content.templates.update', $template->json('id')), [
            'name' => 'foo',
            'slug' => 'foo',
            'description' => 'lorem ipsum',
            'elements' => [
                [
                    'name' => 'Slider',
                    'element_id' => $slider->json('data.id'),
                    'field_type' => 'element',
                    'global_content_id' => null,
                    'description' => 'Slider element',
                    'is_list' => true,
                ],
                [
                    'name' => 'Text Section',
                    'element_id' => $text->json('data.id'),
                    'field_type' => 'text',
                    'global_content_id' => null,
                    'description' => 'Content',
                ],
                [
                    'name' => 'lorem ipsum',
                    'element_id' => null,
                    'field_type' => null,
                    'global_content_id' => $global->json('id'),
                    'description' => 'Global content element',
                ],
            ],
        ]);

        $response->assertStatus(200);

        $page = $this->postJson(route('content.pages.store'), [
            'title' => 'foo baz bar',
            'template_id' => $template->json('id'),
        ]);
        $page->assertStatus(201);
        $page = $this->getJson(route('content.pages.show', $page->json('id')));
        $page->assertStatus(200);

        $this->assertTrue(Content::query()->where('name', 'Text Section')->where('page_id', $page->json('page.id'))->exists());
        $this->assertTrue(Content::query()->where('name', 'lorem ipsum')->where('page_id', $page->json('page.id'))->exists());
        $this->assertTrue(Content::query()->where('name', 'Slider')->where('page_id', $page->json('page.id'))->exists());

        $response = $this->getJson(route('content.pages.show', $page->json('page.id')));
        $response->assertStatus(200);

        $slider_content = $response->json('page.content.0');
        $text_content = $response->json('page.content.1');
        $global_content = $response->json('page.content.2');

        $slider_content['children'][0]['content'] = null;
        $slider_content['children'][1]['content'] = 'Hello World';
        $slider_content['children'][2]['content'] = '/ford/fiesta';

        $text_content['children'][0]['content'] = 'Welcome to foo';
        $text_content['children'][1]['content'] = '<p>lorem ipsum dolor sit amet</p>';

        $response = $this->putJson(route('content.pages.update', $page->json('page.id')), [
            'title' => 'foo baz bar',
            'slug' => 'foo-baz-bar',
            'status' => PageStatus::PUBLISHED->value,
            'seo' => [],
            'meta' => [],
            'content' => [
                $slider_content,
                $text_content,
                $global_content,
            ],
            'template_id' => $template->json('id'),
        ]);
        $response->assertStatus(200);

        $response = $this->getJson(route('content.pages.show', $page->json('page.id')));
        $response->assertStatus(200);

        $this->assertEquals(1, Version::query()->where('page_id', $page->json('page.id'))->count());
    }

    public function testPageSeoData()
    {
        $this->asUser();

        $page = Page::factory()->create();

        $template = Template::factory()->create();
        $response = $this->putJson(route('content.pages.update', $page->id), [
            'title' => 'foo baz bar',
            'slug' => 'foo-baz-bar',
            'template_id' => $template->id,
            'status' => PageStatus::PUBLISHED->value,
            'seo' => [
                'title' => 'foo',
            ],
            'meta' => [],
            'content' => [],
        ]);
        $response->assertStatus(200);
        $this->assertIsArray($response->json('page.seo'));
        $this->assertArrayHasKey('title', $response->json('page.seo'));
        $this->assertEquals('foo', $response->json('page.seo.title'));
        $this->assertArrayHasKey('description', $response->json('page.seo'));
        $this->assertArrayHasKey('heading', $response->json('page.seo'));

        $response = $this->putJson(route('content.pages.update', $page->id), [
            'title' => 'foo baz bar',
            'slug' => 'foo-baz-bar',
            'status' => PageStatus::PUBLISHED->value,
            'seo' => [
                'title' => 'foo',
                'heading' => 'lorem ipsum dolor'
            ],
            'meta' => [],
            'content' => [],
        ]);
        $this->assertEquals('', $response->json('page.seo.description'));
        $this->assertEquals('lorem ipsum dolor', $response->json('page.seo.heading'));
    }

    public function testIncorrectGlobalContent()
    {
        $this->asUser();

        $global_content = GlobalContent::factory()->create();
        $template = Template::factory()->create();
        TemplateElement::factory()->create([
            'template_id' => $template->id,
            'global_content_id' => $global_content->id,
        ]);

        $page = \Mtc\ContentManager\Facades\Page::create('foo', $template->id);
        $page_element = $page->content()->first();

        $response = $this->putJson(route('content.pages.update', $page->id), [
            'title' => 'foo baz bar',
            'slug' => 'foo-baz-bar',
            'status' => PageStatus::PUBLISHED->value,
            'seo' => [],
            'meta' => [],
            'content' => [
                [
                    'id' => $page_element->id,
                    'element_id' => $page_element->element_id,
                    'global_content_id' => 100,
                    'content' => [],
                ],
            ],
            'template_id' => $template->id,
        ]);
        $response->assertStatus(422);

        $this->assertArrayHasKey('content.0', $response->json('errors'));
    }

    public function testIncorrectContentElement()
    {
        $this->asUser();

        $element = ContentElement::factory()->create();
        ContentElementField::factory()->create(['element_id' => $element->id]);
        $template = Template::factory()->create();
        TemplateElement::factory()->create([
            'template_id' => $template->id,
            'element_id' => $element->id,
        ]);

        $page = \Mtc\ContentManager\Facades\Page::create('foo', $template->id);
        $page_element = $page->content()->first();

        $response = $this->putJson(route('content.pages.update', $page->id), [
            'title' => 'foo baz bar',
            'slug' => 'foo-baz-bar',
            'status' => PageStatus::PUBLISHED->value,
            'seo' => [],
            'meta' => [],
            'content' => [
                [
                    'id' => $page_element->id,
                    'element_id' => $page_element->element_id,
                    'global_content_id' => $page_element->global_content_id,
                    'content' => [],
                ],
            ],
            'template_id' => $template->id,
        ]);
        $response->assertStatus(422);

        $this->assertArrayHasKey('content.0', $response->json('errors'));
    }

    public function testCopyPageNoContent()
    {
        $this->asUser();
        $slider = $this->postJson(route('content.elements.store'), ['title' => 'Slider']);
        $slider->assertStatus(201);

        $response = $this->putJson(route('content.elements.update', $slider->json('data.id')), [
            'title' => 'Slider',
            'subtitle' => 'baz',
            'category' => 'bar',
            'icon' => '/path/to/icon',
            'is_enabled' => true,
            'fields' => [
                [
                    'id' => 'test-1',
                    'name' => 'Image',
                    'field_type' => 'image'
                ],
                [
                    'id' => 'test-2',
                    'name' => 'Title',
                    'field_type' => 'text'
                ],
                [
                    'id' => 'test-3',
                    'name' => 'Link',
                    'field_type' => 'text'
                ],
            ]
        ]);
        $response->assertStatus(200);

        $text = $this->postJson(route('content.elements.store'), ['title' => 'Text Block']);
        $text->assertStatus(201);

        $response = $this->putJson(route('content.elements.update', $text->json('data.id')), [
            'title' => 'Text Block',
            'subtitle' => 'baz',
            'category' => 'bar',
            'icon' => '/path/to/icon',
            'is_enabled' => true,
            'fields' => [
                [
                    'id' => 'test-1',
                    'name' => 'Title',
                    'field_type' => 'text'
                ],
                [
                    'id' => 'test-2',
                    'name' => 'Text',
                    'field_type' => 'textarea'
                ],
            ]
        ]);
        $response->assertStatus(200);

        $global = $this->postJson(route('content.global-elements.store'), [
            'name' => 'lorem ipsum',
            'element_id' => $text->json('data.id'),
        ]);
        $global->assertStatus(201);

        $template = $this->postJson(route('content.templates.store'), ['name' => 'Default Template']);
        $template->assertStatus(201);

        $templateResponse = $this->putJson(route('content.templates.update', $template->json('id')), [
            'name' => 'foo',
            'slug' => 'foo',
            'description' => 'lorem ipsum',
            'elements' => [
                [
                    'name' => 'Slider',
                    'element_id' => $slider->json('data.id'),
                    'field_type' => 'element',
                    'global_content_id' => null,
                    'description' => 'Slider element',
                    'is_list' => true,
                ],
                [
                    'name' => 'Text Section',
                    'element_id' => $text->json('data.id'),
                    'field_type' => 'text',
                    'global_content_id' => null,
                    'description' => 'Content',
                ],
                [
                    'name' => 'lorem ipsum',
                    'element_id' => null,
                    'field_type' => null,
                    'global_content_id' => $global->json('id'),
                    'description' => 'Global content element',
                ],
            ],
        ]);

        $templateResponse->assertStatus(200);

        $page = $this->postJson(route('content.pages.store'), [
            'title' => 'foo baz bar',
            'template_id' => $template->json('id'),
        ]);
        $page->assertStatus(201);

        $page = $this->getJson(route('content.pages.show', $page->json('id')));
        $page->assertStatus(200);

        $this->assertTrue(Content::query()->where('name', 'Text Section')->where('page_id', $page->json('page.id'))->exists());
        $this->assertTrue(Content::query()->where('name', 'lorem ipsum')->where('page_id', $page->json('page.id'))->exists());
        $this->assertTrue(Content::query()->where('name', 'Slider')->where('page_id', $page->json('page.id'))->exists());

        $response = $this->getJson(route('content.pages.show', $page->json('page.id')));
        $response->assertStatus(200);

        $slider_content = $response->json('page.content.0');
        $text_content = $response->json('page.content.1');
        $global_content = $response->json('page.content.2');

        $slider_content['children'][0]['content'] = null;
        $slider_content['children'][1]['content'] = 'Hello World';
        $slider_content['children'][2]['content'] = '/ford/fiesta';

        $text_content['children'][0]['content'] = 'Welcome to foo';
        $text_content['children'][1]['content'] = '<p>lorem ipsum dolor sit amet</p>';


        $response = $this->putJson(route('content.pages.update', $page->json('page.id')), [
            'title' => 'foo baz bar',
            'slug' => 'foo-baz-bar',
            'status' => PageStatus::PUBLISHED->value,
            'seo' => [],
            'meta' => [],
            'content' => [
                $slider_content,
                $text_content,
                $global_content
            ],
            'template_id' => $template->json('id'),
        ]);
        $response->assertStatus(200);

        $response = $this->postJson(route('content.pages.copy', $page->json('page.id')), [
            'title' => 'foo baz bar',
            'with_content' => false,
        ]);
        $response->assertStatus(201);
        $pageId = $response->json('page.id');
    }

    public function testCopyPageWithContent()
    {
        $this->asUser();
        $slider = $this->postJson(route('content.elements.store'), ['title' => 'Slider']);
        $slider->assertStatus(201);

        $response = $this->putJson(route('content.elements.update', $slider->json('data.id')), [
            'title' => 'Slider',
            'subtitle' => 'baz',
            'category' => 'bar',
            'icon' => '/path/to/icon',
            'is_enabled' => true,
            'fields' => [
                [
                    'id' => 'test-1',
                    'name' => 'Image',
                    'field_type' => 'image'
                ],
                [
                    'id' => 'test-2',
                    'name' => 'Title',
                    'field_type' => 'text'
                ],
                [
                    'id' => 'test-3',
                    'name' => 'Link',
                    'field_type' => 'text'
                ],
            ]
        ]);
        $response->assertStatus(200);

        $text = $this->postJson(route('content.elements.store'), ['title' => 'Text Block']);
        $text->assertStatus(201);

        $response = $this->putJson(route('content.elements.update', $text->json('data.id')), [
            'title' => 'Text Block',
            'subtitle' => 'baz',
            'category' => 'bar',
            'icon' => '/path/to/icon',
            'is_enabled' => true,
            'fields' => [
                [
                    'id' => 'test-1',
                    'name' => 'Title',
                    'field_type' => 'text'
                ],
                [
                    'id' => 'test-2',
                    'name' => 'Text',
                    'field_type' => 'textarea'
                ],
            ]
        ]);
        $response->assertStatus(200);

        $global = $this->postJson(route('content.global-elements.store'), [
            'name' => 'lorem ipsum',
            'element_id' => $text->json('data.id'),
        ]);
        $global->assertStatus(201);

        $template = $this->postJson(route('content.templates.store'), ['name' => 'Default Template']);
        $template->assertStatus(201);

        $templateResponse = $this->putJson(route('content.templates.update', $template->json('id')), [
            'name' => 'foo',
            'slug' => 'foo',
            'description' => 'lorem ipsum',
            'elements' => [
                [
                    'name' => 'Slider',
                    'element_id' => $slider->json('data.id'),
                    'field_type' => 'element',
                    'global_content_id' => null,
                    'description' => 'Slider element',
                    'is_list' => true,
                ],
                [
                    'name' => 'Text Section',
                    'element_id' => $text->json('data.id'),
                    'field_type' => 'text',
                    'global_content_id' => null,
                    'description' => 'Content',
                ],
                [
                    'name' => 'lorem ipsum',
                    'element_id' => null,
                    'field_type' => null,
                    'global_content_id' => $global->json('id'),
                    'description' => 'Global content element',
                ],
            ],
        ]);

        $templateResponse->assertStatus(200);

        $page = $this->postJson(route('content.pages.store'), [
            'title' => 'foo baz bar',
            'template_id' => $template->json('id'),
        ]);
        $page->assertStatus(201);
        $page = $this->getJson(route('content.pages.show', $page->json('id')));
        $page->assertStatus(200);

        $this->assertTrue(Content::query()->where('name', 'Text Section')->where('page_id', $page->json('page.id'))->exists());
        $this->assertTrue(Content::query()->where('name', 'lorem ipsum')->where('page_id', $page->json('page.id'))->exists());
        $this->assertTrue(Content::query()->where('name', 'Slider')->where('page_id', $page->json('page.id'))->exists());

        $response = $this->getJson(route('content.pages.show', $page->json('page.id')));
        $response->assertStatus(200);

        $slider_content = $response->json('page.content.0');
        $text_content = $response->json('page.content.1');
        $global_content = $response->json('page.content.2');

        $slider_content['children'][0]['content'] = null;
        $slider_content['children'][1]['content'] = 'Hello World';
        $slider_content['children'][2]['content'] = '/ford/fiesta';

        $text_content['children'][0]['content'] = 'Welcome to foo';
        $text_content['children'][1]['content'] = '<p>lorem ipsum dolor sit amet</p>';


        $response = $this->putJson(route('content.pages.update', $page->json('page.id')), [
            'title' => 'foo baz bar',
            'slug' => 'foo-baz-bar',
            'status' => PageStatus::PUBLISHED->value,
            'seo' => [],
            'meta' => [],
            'content' => [
                $slider_content,
                $text_content,
                $global_content
            ],
            'template_id' => $template->json('id'),
        ]);
        $response->assertStatus(200);

        $response = $this->postJson(route('content.pages.copy', $page->json('page.id')), [
            'title' => 'foo baz bar',
            'with_content' => true,
        ]);
        $response->assertStatus(201);
    }
}
