<?php

namespace Mtc\ContentManager\Tests\Unit;

use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Notification;
use Mtc\ContentManager\Facades\Comment;
use Mtc\ContentManager\Models\Comment as CommentModel;
use Mtc\ContentManager\Models\Page;
use Mtc\ContentManager\Notifications\NewCommentNotification;
use Mtc\ContentManager\Tests\TestCase;
use Mtc\ContentManager\Tests\User;

class CommentRepositoryTest extends TestCase
{
    use RefreshDatabase;

    public function testAddComment(): void
    {
        /** @var Page $page */
        $page = Page::factory()->create();
        $user = User::query()->create(['name' => 'john', 'email' => 'hello@example.com', 'password' => '']);

        $comment = Comment::add('lorem-ipsum', $user->id, $page);

        $this->assertEquals(1, $page->comments()->count());
        $this->assertInstanceOf(CommentModel::class, $comment);
    }

    public function testAddReplyComment()
    {
        Notification::fake();
        /** @var Page $page */
        $page = Page::factory()->create();
        $user = User::query()->create(['name' => 'john', 'email' => 'hello@example.com', 'password' => '']);
        $user2 = User::query()->create(['name' => 'john', 'email' => 'foo@example.com', 'password' => '']);

        $comment = Comment::add('lorem-ipsum', $user->id, $page);
        $reply = Comment::add('foo-baz-bar', $user2->id, $page, $comment->id);

        $this->assertEquals(2, $page->comments()->count());
        $this->assertInstanceOf(CommentModel::class, $reply);

        Notification::assertSentTo($user, NewCommentNotification::class);
    }

    public function testNotificationSentToTaggedUser()
    {
        Notification::fake();
        /** @var Page $page */
        $page = Page::factory()->create();
        $user = User::query()->create(['name' => 'john', 'email' => 'hello@example.com', 'password' => '']);
        $user2 = User::query()->create(['name' => 'john', 'email' => 'foo@example.com', 'password' => '']);

        Comment::add('lorem-ipsum', $user->id, $page, null, [$user2->id]);

        Notification::assertSentTo($user2, NewCommentNotification::class);
    }

    public function testTaggedUserIsPartOfNotification()
    {
        Notification::fake();
        /** @var Page $page */
        $page = Page::factory()->create();
        $user = User::query()->create(['name' => 'john', 'email' => 'hello@example.com', 'password' => '']);
        $user2 = User::query()->create(['name' => 'john', 'email' => 'foo@example.com', 'password' => '']);

        $comment = Comment::add('lorem-ipsum', $user->id, $page, null, [$user2->id]);

        $comments = Comment::getForModel($page);
        $this->assertInstanceOf(Collection::class, $comments);
        $this->assertEquals(1, $comments->count());
        $this->assertTrue($comments->first()->is($comment));
        $this->assertTrue($comments->first()->taggedUsers->find($user2->id)->exists);
    }

    public function testGetForModel(): void
    {
        /** @var Page $page */
        $page = Page::factory()->create();
        $user = User::query()->create(['name' => 'john', 'email' => 'hello@example.com', 'password' => '']);

        Comment::add('lorem-ipsum', $user->id, $page);

        $comments = Comment::getForModel($page);

        $this->assertInstanceOf(Collection::class, $comments);
        $this->assertEquals(1, $comments->count());
        $this->assertInstanceOf(CommentModel::class, $comments->first());
    }

    public function testMarkResolved(): void
    {
        /** @var Page $page */
        $page = Page::factory()->create();
        $user = User::query()->create(['name' => 'john', 'email' => 'hello@example.com', 'password' => '']);

        $comment = Comment::add('lorem-ipsum', $user->id, $page);

        $this->assertFalse($comment->isResolved());
        Comment::markResolved($comment->id);

        $comment->refresh();
        $this->assertTrue($comment->isResolved());
    }
}
