<?php

namespace Mtc\ContentManager\Contracts;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Mtc\ContentManager\Factories\MediaFactory;
use Mtc\ContentManager\ImageSize;
use Symfony\Component\HttpKernel\Exception\NotFoundHttpException;

abstract class Media extends Model
{
    use HasFactory;

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'type',
        'src',
        'path',
        'alt_text',
        'title',
        'caption',
        'description',
    ];

    /**
     * Hide attributes from json
     *
     * @var string[]
     */
    protected $hidden = [
        'created_at',
        'updated_at'
    ];

    protected $appends = [
        'original_url',
        'square_thumb_url',
        'resizable',
    ];

    /**
     * Cast attributes to types
     *
     * @var string[]
     */
    protected $casts = [
        'meta' => 'array',
    ];

    /**
     * Model Factory
     *
     * @return MediaFactory
     */
    protected static function newFactory()
    {
        return MediaFactory::new();
    }

    protected static function boot()
    {
        parent::boot();

        self::creating(static fn($self) => $self->upload_date = now()->format('Y-m'));
    }

    /**
     * @return HasMany
     */
    public function uses(): HasMany
    {
        return $this->hasMany(Config::get('media.media_use_model'), 'media_id');
    }


    /**
     * @return HasMany
     */
    public function tags(): HasMany
    {
        return $this->hasMany(Config::get('media.tag_model'), 'media_id');
    }

    /**
     * Add usage_count to query
     *
     * @param Builder $query
     */
    public function scopeWithUsageCount(Builder $query)
    {
        $query->addSelect([
            'usage_count' => function ($query) {
                $query->selectRaw(DB::raw('count(*)'))
                    ->from('media_uses')
                    ->whereColumn('media_id', 'media.id')
                    ->limit(1);
            }
        ]);
    }

    /**
     * check if media has resizable file type
     *
     * @return bool
     */
    public function resizableMime()
    {
        return $this->type === 'image';
    }

    /**
     * get resizable attribute
     *
     * @return bool
     */
    public function getResizableAttribute(): bool
    {
        return $this->resizableMime();
    }

    /**
     * square_thumb attribute
     *
     * @return string
     */
    public function getSquareThumbUrlAttribute(): string
    {
        $size = ImageSize::fromArray(Config::get('media.default_thumbnail_size'));
        return Storage::disk(Config::get('filesystems.default_media'))->url($size->pathOnDisk($this->src));
    }

    /**
     * original_url attribute (url to the original size)
     *
     * @return string
     */
    public function getOriginalUrlAttribute(): string
    {
        return Storage::disk(Config::get('filesystems.default_media'))->url($this->path . '/' . $this->src);
    }

    /**
     * @return string
     */
    public function getOriginalFilePath(): string
    {
        return $this->path . '/' . $this->src;
    }

    /**
     * @return string
     */
    public function thumbnailFilePath(): string
    {
        return ImageSize::fromArray(Config::get('media.default_thumbnail_size'))->pathOnDisk($this->src);
    }
}
