<?php

namespace Mtc\ContentManager\Http\Controllers;

use Illuminate\Foundation\Validation\ValidatesRequests;
use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Illuminate\Support\Facades\Config;
use Mtc\ContentManager\Contracts\Media as MediaModel;
use Mtc\ContentManager\Facades\Media;
use Mtc\ContentManager\Http\Requests\MediaResizeRequest;
use Mtc\ContentManager\Http\Requests\MediaUpdateRequest;
use Mtc\ContentManager\Http\Requests\MediaUploadRequest;
use Mtc\ContentManager\Http\Resources\MediaFileResource;
use Mtc\ContentManager\Http\Resources\MediaListResource;

class MediaController
{
    use ValidatesRequests;

    /**
     * List media sizes and objects for model
     *
     * @return MediaListResource
     */
    public function index(Request $request): MediaListResource
    {
        $resource = Config::get('pages.media_list_resource');
        return new $resource(Media::getList($request));
    }

    /**
     * Process file upload
     *
     * @param Request $request
     * @return mixed
     */
    public function show(int $mediaId)
    {
        $resource = Config::get('pages.media_file_resource');
        return new $resource(Media::find($mediaId));
    }
    /**
     * Process file upload
     *
     * @param Request $request
     * @return mixed
     */
    public function store(MediaUploadRequest $request)
    {
        $resource = Config::get('pages.media_file_resource');
        return new $resource(Media::upload($request));
    }

    /**
     * Store request is used for storing a new image version
     * This is used by the image crop tool which sends the new coordinates for image size
     * Method finds the media record and attempts cropping the original image to specified size
     *
     * @param Request $request incoming request
     * @return array
     */
    public function resize(MediaResizeRequest $request)
    {
        return [
            'url' => Media::resize($request, $request->input('coordinates'))
        ];
    }

    /**
     * Remove the media object
     *
     * @param int $mediaId
     * @return Response
     */
    public function destroy(int $mediaId): Response
    {
        Media::destroyMedia($mediaId);
        return response('success', 200);
    }

    /**
     * Update meta data of media object
     *
     * @param MediaUpdateRequest $request
     * @param MediaModel $media
     */
    public function update(MediaUpdateRequest $request, MediaModel $medium)
    {
        $medium->update([
            'alt_text' => $request->input('alt_text'),
            'title' => $request->input('title'),
            'caption' => $request->input('caption'),
            'description' => $request->input('description'),
        ]);

        $resource = Config::get('pages.media_file_resource');
        return new $resource($medium->refresh());
    }
}
