<?php

namespace Mtc\ContentManager\Rules;

use Illuminate\Contracts\Validation\Rule;
use Illuminate\Support\Facades\App;
use Mtc\ContentManager\Contracts\ContentElement;
use Mtc\ContentManager\Contracts\ContentElementField;
use Mtc\ContentManager\Contracts\GlobalContent;

class PageContentHasAllBlueprintFields implements Rule
{
    private GlobalContent $global_content;
    private ContentElement $element;
    /**
     * Create a new rule instance.
     *
     * @return void
     */
    public function __construct()
    {
        $this->global_content = App::make(GlobalContent::class);
        $this->element = App::make(ContentElement::class);
    }

    /**
     * Determine if the validation rule passes.
     *
     * @param  string  $attribute
     * @param  mixed  $value
     * @return bool
     */
    public function passes($attribute, $value)
    {
        if (!empty($value['global_content_id'])) {
            return $this->passesForGlobalContent($value['global_content_id'], $value['content']);
        }
        return $this->passesForElement($value['element_id'] ?? 0, $value['content']);
    }

    /**
     * Get the validation error message.
     *
     * @return string
     */
    public function message()
    {
        return 'The content of element does not have all blueprint fields.';
    }

    /**
     * Verify that content is valid for global content element
     *
     * @param $elementId
     * @param $value
     * @return bool
     */
    protected function passesForGlobalContent($elementId, $value): bool
    {
        // Content is not relevant as global content defines content
        // We only need to make sure that the global content element exists
        return $this->global_content->newQuery()->where('id', $elementId)->exists();
    }

    /**
     * Verify that content element contains all data based on the element blueprint
     *
     * @param $elementId
     * @param $value
     * @return bool
     */
    protected function passesForElement($elementId, $value): bool
    {
        /** @var ContentElement|null $blueprint */
        $blueprint = $this->element->newQuery()->where('is_enabled', true)->find($elementId);
        if ($blueprint === null) {
            return false;
        }

        // Check if fields are missing
        return $blueprint->fields()->get()
            ->reject(fn (ContentElementField $field) => array_key_exists($field->name, $value))
            ->isEmpty();
    }
}
