<?php

namespace Mtc\ContentManager\Http\Controllers;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Foundation\Validation\ValidatesRequests;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\DB;
use Mtc\ContentManager\CommentRepository;
use Mtc\ContentManager\Contracts\Template;
use Mtc\ContentManager\Contracts\TemplateElement;
use Mtc\ContentManager\ElementRepository;
use Mtc\ContentManager\Http\Requests\CopyTemplateRequest;
use Mtc\ContentManager\Http\Requests\StoreTemplateRequest;
use Mtc\ContentManager\Http\Requests\UpdateTemplateRequest;
use Mtc\ContentManager\Http\Resources\TemplateListResource;

class TemplateController
{
    use ValidatesRequests;

    /**
     * List available template records
     *
     * @param Request $request
     * @param Template $layout
     * @return TemplateListResource
     */
    public function index(Request $request, Template $layout)
    {
        $results = $layout->newQuery()
            ->orderByDesc($request->input('sortBy', 'updated_at'))
            ->withInactivePageCount()
            ->withCommentCount()
            ->withActivePageCount()
            ->with('updatedBy')
            ->paginate();

        $resourceClass = Config::get('pages.template_list_resource');
        return new $resourceClass($results);
    }

    /**
     * Store a new template record
     *
     * @param StoreTemplateRequest $request
     * @param Template $template
     * @return Template|Model
     * @throws \Illuminate\Validation\ValidationException
     */
    public function store(StoreTemplateRequest $request, Template $template)
    {
        return $template->newQuery()
            ->create([
                'name' => $request->input('name'),
            ]);
    }

    /**
     * Show details of template
     *
     * @param Template $template
     * @param ElementRepository $repository
     * @param CommentRepository $comments
     * @return array
     */
    public function show(Template $template, ElementRepository $repository, CommentRepository $comments)
    {
        return [
            'template' => $template,
            'template_elements' => $template->elements,
            'elements' => $repository->getList(),
            'comments' => $comments->getForModel($template)
        ];
    }

    /**
     * Update the template record
     *
     * @param Request $request
     * @param Template $template
     * @return array
     */
    public function update(UpdateTemplateRequest $request, Template $template, TemplateElement $templateElement)
    {
        if (config('pages.use_transactions')) {
            DB::beginTransaction();
        }
        $template->fill($request->input())->save();
        $template->elements()
            ->whereNotIn('id', collect($request->input('elements'))->pluck('id'))
            ->delete();

        collect($request->input('elements'))
            ->each(function ($element, $order) use ($template, $templateElement) {
                /** @var TemplateElement $data */
                $data = $templateElement->newQuery()->findOrNew($element['id'] ?? 0);
                if ($data->exists === false) {
                    $data->template_id = $template->id;
                }

                $data->fill([
                    'name' => $element['name'],
                    'global_content_id' => $element['global_content_id'] ?? null,
                    'element_id' => $element['element_id'] ?? null,
                    'description' => $element['description'] ?? '',
                    'hint' => $element['hint'] ?? '',
                    'is_list' => $element['is_list'] ?? false,
                    'max_repeat_count' => $element['max_repeat_count'] ?? false,
                    'order' => $order,
                ]);

                $data->save();
            });

        if (config('pages.use_transactions')) {
            DB::commit();
        }

        $template->refresh();
        $template->load('elements');
        return [
            'template' => $template,
            'template_elements' => $template->elements,
            'success' => true
        ];
    }

    /**
     * Remove Template
     *
     * @param Template $template
     * @return bool|null
     */
    public function destroy(Template $template)
    {
        return $template->delete();
    }

    /**
     * List unused templates (safe-to-remove)
     *
     * @param Request $request
     * @param Template $template
     */
    public function unused(Template $template): TemplateListResource
    {
        $templates = $template->newQuery()
            ->whereDoesntHave('pages')
            ->paginate();

        return new TemplateListResource($templates);
    }


    /**
     * Create a new copy of an existing global content element
     *
     * @param CopyTemplateRequest $request
     * @param int $id_to_copy
     * @return Template|Model
     */
    public function copy(CopyTemplateRequest $request, Template $template): Template
    {
        /** @var Template $copy */
        $copy = $template->newQuery()->create([
            'name' => $request->input('name'),
            'description' => $template->description,
            'seo_defaults' => $template->seo_defaults,
        ]);

        $template->elements
            ->each(fn(TemplateElement $original) => $copy->elements()->create($original->toArray()));

        return $copy;
    }
}
