<?php

namespace Mtc\Core\Http\Controllers\Admin;

use Illuminate\Http\Request;
use Mtc\Core\Http\Controllers\Controller;
use Mtc\Core\Seo\Heading;
use Mtc\Core\Seo\Page;

/**
 * Class SeoHeadingController
 *
 * @package Mtc\Core\Http\Controllers\Admin
 */
class SeoHeadingController extends Controller
{
    /**
     * Displays Seo Heading list
     *
     * @return string
     * @throws \Twig_Error_Loader
     * @throws \Twig_Error_Runtime
     * @throws \Twig_Error_Syntax
     */
    public function index()
    {
        $this->page_meta['title'] = 'Manage SEO Headings';

        return template('admin/seo/headings/index.twig', [
            'page_meta' => $this->page_meta,
            'headings' => Heading::query()->orderBy('path')->get(),
        ]);
    }

    /**
     * Render view for creating a heading
     *
     * @return string
     * @throws \Twig_Error_Loader
     * @throws \Twig_Error_Runtime
     * @throws \Twig_Error_Syntax
     */
    public function create()
    {
        $this->page_meta['title'] = 'Add Seo Heading';
        return template('admin/seo/headings/edit.twig', [
            'page_meta' => $this->page_meta,
            'heading' => new Heading(),
        ]);

    }

    /**
     * Store a seo heading
     *
     * @param Request $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function store(Request $request)
    {
        $validated = $this->validate($request, [
            'path' => 'required|unique:seo_headings,path',
            'text' => 'required',
        ]);

        Heading::query()->create($validated);
        session()->flash('success', 'Seo Page entry Created');
        return redirect()->to("/admin/seo/headings/");
    }

    /**
     * Render view for editing a heading
     *
     * @param $heading_id
     * @return string
     * @throws \Twig_Error_Loader
     * @throws \Twig_Error_Runtime
     * @throws \Twig_Error_Syntax
     */
    public function edit($heading_id)
    {
        $heading = Heading::query()->find($heading_id);
        $this->page_meta['title'] = 'Edit Seo Heading';
        return template('admin/seo/headings/edit.twig', [
            'page_meta' => $this->page_meta,
            'heading' => $heading,
        ]);
    }

    /**
     * Update a seo heading
     *
     * @param Request $request
     * @param $heading_id
     * @return \Illuminate\Http\RedirectResponse
     */
    public function update(Request $request, $heading_id)
    {
        $validated = $this->validate($request, [
            'text' => 'required',
        ]);

        $heading = Heading::query()->find($heading_id);
        $heading->fill($validated)->save();
        session()->flash('success', 'Seo Heading entry Updated');
        return redirect()->to("/admin/seo/headings/");
    }

    /**
     * Delete seo heading entry
     *
     * @param int $heading_id
     */
    public function destroy($heading_id)
    {
        $heading = Page::query()->findOrFail($heading_id);
        $heading->delete();
        session()->flash('success', 'Seo Heading Entry Removed');
        return redirect('/admin/seo/pages');
    }
}
