<?php

namespace Mtc\Core\Http\Controllers\Admin;

use Illuminate\Http\Request;
use Mtc\Core\Http\Controllers\Controller;
use Mtc\Core\Seo\Page;

/**
 * Class SeoPageController
 *
 * @package Mtc\Core\Http\Controllers\Admin
 */
class SeoPageController extends Controller
{
    /**
     * Displays Seo Page list
     *
     * @return string
     * @throws \Twig_Error_Loader
     * @throws \Twig_Error_Runtime
     * @throws \Twig_Error_Syntax
     */
    public function index()
    {
        $this->page_meta['title'] = 'Manage SEO Pages';

        return template('admin/seo/pages/index.twig', [
            'page_meta' => $this->page_meta,
            'pages' => Page::query()->orderBy('path')->get(),
        ]);
    }

    /**
     * Render view for creating a page
     *
     * @return string
     * @throws \Twig_Error_Loader
     * @throws \Twig_Error_Runtime
     * @throws \Twig_Error_Syntax
     */
    public function create()
    {
        $this->page_meta['title'] = 'Add Seo Page';
        return template('admin/seo/pages/edit.twig', [
            'page_meta' => $this->page_meta,
            'page' => Page::query()->orderBy('path')->get(),
        ]);

    }

    /**
     * Store a seo page
     *
     * @param Request $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function store(Request $request)
    {
        $validated = $this->validate($request, [
            'path' => 'required|unique:seo_pages,path',
            'title' => 'required',
            'description' => 'required',
        ]);

        $page = Page::query()->create($validated);
        session()->flash('success', 'Seo Page entry Created');
        return redirect()->to("/admin/seo/pages/");
    }

    /**
     * Render view for editing a page
     *
     * @param $page_id
     * @return string
     * @throws \Twig_Error_Loader
     * @throws \Twig_Error_Runtime
     * @throws \Twig_Error_Syntax
     */
    public function edit($page_id)
    {
        $page = Page::query()->find($page_id);
        $this->page_meta['title'] = 'Edit Seo Page';
        return template('admin/seo/pages/edit.twig', [
            'page_meta' => $this->page_meta,
            'page' => $page,
        ]);
    }

    /**
     * Update a seo page
     *
     * @param Request $request
     * @param $page_id
     * @return \Illuminate\Http\RedirectResponse
     */
    public function update(Request $request, $page_id)
    {
        $validated = $this->validate($request, [
            'title' => 'required',
            'description' => 'required',
        ]);

        $page = Page::query()->find($page_id);
        $page->fill($validated)->save();
        session()->flash('success', 'Seo Page entry Updated');
        return redirect()->to("/admin/seo/pages/");
    }

    /**
     * Delete seo page entry
     *
     * @param int $page_id
     */
    public function destroy($page_id)
    {
        $page = Page::query()->findOrFail($page_id);
        $page->delete();
        session()->flash('success', 'Seo Page Entry Removed');
        return redirect('/admin/seo/pages');
    }
}
