<?php
/**
 * Currency Model
 *
 * Deals with the update and resolve of currencies.
 *
 * @author Andrew Morgan <andrew.morgan@mtcmedia.co.uk>
 */

namespace Mtc\Core;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Cache;
use Mtc\Core\Models\Country;

/**
 * Currency Model
 *
 * @author Andrew Morgan <andrew.morgan@mtcmedia.co.uk>
 */
class Currency extends Model
{
    /**
     * @var string Model table name
     */
    protected $table = 'currencies';

    /**
     * @var array The attributes that are mass assignable.
     */
    protected $fillable = [
        'currency',
        'ratio',
        'enabled',
    ];

    /**
     * All currency rates will be stored relative to this one. Table will auto
     * update if changed during active session.
     * @var string
     */
    const SITE_CURRENCY = 'GBP';

    /**
     * URL for currency feed (serves rates in euro)
     * @var string
     */
    private static $currency_feed_address = 'http://www.ecb.europa.eu/stats/eurofxref/eurofxref-daily.xml';

    /**
     * Minimum number of currencies expected from feed. There were 33 at the
     * time of writing (1/4/14) but I thought a lower limit would be better in
     * case the number ever drops.
     *
     * The limit of 10 should avoid processing null results but not require a
     * global update if the ECB decides to change the feed.
     *
     * @var int
     */
    private static $min_num_currencies = 10;

    /**
     * Determines if DEV_EMAIL should recieve currency error reports
     * @var boolean
     */
    private static $notify_dev = true;

    /**
     * Currency rates relative to the Euro
     * @var array
     */
    private $euro_rates = [];

    /**
     * Currency rates relative to the British Pound
     * @var array
     */
    private $site_rates = [];

    protected static function boot()
    {
        parent::boot();

        self::saved(function ($currency) {
            Cache::forget('enabled-currencies');
        });
    }

    /**
     * Scope - active()
     *
     * @param \Illuminate\Database\Eloquent\Builder $query Query builder object
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public static function scopeActive($query)
    {
        return $query->where('enabled', 1);
    }

    /**
     * getEnabledCurrencies()
     * Return all enabled currencies
     *
     * @return array Enabled currencies keyed by currency code (GBP etc.)
     */
    public static function getEnabledCurrencies()
    {
        return Cache::remember('enabled-currencies', now()->addMinutes(10), function () {
            return self::active()
                ->when(config('settings.ENABLE_CURRENCIES') !== true, function ($query) {
                    return $query->where('currency', self::SITE_CURRENCY);
                })
                ->get()
                ->keyBy('currency');
        });
    }

    /**
     * Return the currency associated with a particular country code
     *
     * @param string the 2 digit code of a particular country
     * @return string currency code for the matched country
     */
    public static function currencyFromCountryCode($country_code)
    {
        $country_currency = Country::where("code", $country_code)
                                ->value("currency_code");

        if (empty($country_currency)) {
            $country_currency = self::SITE_CURRENCY;
        }

        return $country_currency;
    }


    /**
     * wrapper method calling methods needed to perform cron update
     */
    public function updateCurrencyTable()
    {
        $this->getRatesFromRemoteFeed();
        $this->calculateSiteRates();
        $this->saveSiteRates();
    }

    /**
     * Connects to ECB feed to fetch euro rates. Based on code provided here:
     * http://www.ecb.europa.eu/stats/exchange/eurofxref/html/index.en.html
     *
     * @return boolean
     */
    private function getRatesFromRemoteFeed()
    {
        $ctx = stream_context_create([
            'http' => [
                'timeout' => 1
            ]
        ]);
        $XML_content = file(self::$currency_feed_address);
        if (is_array($XML_content) && !empty($XML_content)) {
            $this->euro_rates['EUR'] = 1;
            foreach ($XML_content as $line) {
                if (preg_match("/currency='([[:alpha:]]+)'/", $line, $currency_code)) {
                    if (preg_match("/rate='([[:graph:]]+)'/", $line, $rate)) {
                        $this->euro_rates[$currency_code[1]] = $rate[1];
                    }
                }
            }
            return true;
        }

        if (self::$notify_dev) {
            email(DEV_EMAIL, 'ERROR: ' . SITE_NAME . ' currency issue', SITE_NAME
                . ' currency cron job failed to connect to ECB at '
                . date('Y-m-d H:i:s')
            );
            return false;
        }
    }

    /**
     * Takes euro rates and loops them against self::SITE_CURRENCY to eur rate
     * to determine self::SITE_CURRENCY
     * rates.
     *
     * @return boolean
     */
    private function calculateSiteRates()
    {
        if (isset($this->euro_rates[self::SITE_CURRENCY])) {
            $site_rate = $this->euro_rates[self::SITE_CURRENCY];
            foreach ($this->euro_rates as $currency => $rate) {
                $this->site_rates[$currency] = round($rate / $site_rate, 4);
            }
            return true;
        }

        if (self::$notify_dev) {
            email(DEV_EMAIL, 'ERROR: ' . SITE_NAME . ' currency issue', SITE_NAME
                . ' site currency not included in xml feed currencies at '
                . date('Y-m-d H:i:s')
            );
            return false;
        }
    }

    /**
     * Updates currencies table from self::SITE_CURRENCY rates
     *
     * @return boolean
     */
    private function saveSiteRates()
    {
        if (count($this->site_rates) >= self::$min_num_currencies
            && isset($this->site_rates[self::SITE_CURRENCY]))
        {
            foreach ($this->site_rates as $currency => $ratio) {
                $record = self::firstOrNew([
                    'currency' => $currency
                ]);
                $record->ratio = $ratio;
                $record->save();
            }
            return true;
        }

        if (self::$notify_dev) {
            email(DEV_EMAIL, 'ERROR: ' . SITE_NAME
                . ' currency issue', SITE_NAME
                . ' insuffiecient currencies fetched at '
                . date('Y-m-d H:i:s')
            );
            return false;
        }
    }

    /**
     * React to user requesting change of the selected currency
     *
     * @return boolean
     */
    public static function changeCurrency($request)
    {
        // Change currency if requested
        if (!empty($request['action'])
            && $request['action'] == 'changecurrency')
        {
            if (!empty($request['currency'])) {
                if (in_array(
                    strip_tags($request['currency']),
                    array_keys(self::getEnabledCurrencies()->toArray()))
                ) {
                    // Store a cookie as user choice
                    $_SESSION['selected_currency'] = $request['currency'];
                    setcookie(
                        'selected_currency',
                        $request['currency'],
                        time() + (2600 * 24 * 30 * 6),
                        "/"
                    );
                }
            }
        }
    }

    /**
     * Add currency type & settings to session variable if found, else 
     * $_SESSION['currency'] will not be set. Units wishing to access this 
     * variable should check to ensure that this variable is set and not null
     *
     * @return array|void depending on whether or not the requested currency
     * was found or not
     */
    public static function getCurrency()
    {

        if (!empty($_COOKIE['selected_currency'])) {
            // Set the currency based on previously selected choice by client
            $country_currency = $_COOKIE['selected_currency'];
        } elseif (!empty($_SESSION['selected_currency'])) {
            // If currency has been previously set... use that
            $country_currency = $_SESSION['selected_currency'];
        } elseif (!empty($_SESSION['detected_country'])) {
            // If no currency is already in place... try and detect from country
            $country_currency = self::currencyFromCountryCode($_SESSION['detected_country']);
        } else {
            // Default currency selected
            $country_currency = self::SITE_CURRENCY;
        }

        if ($_SESSION['selected_currency'] === $country_currency && !empty($_SESSION['currency'])) {
            return $_SESSION['currency'];
        }

        // Store a cookie and session for user choice
        $_SESSION['selected_currency'] = $country_currency;

        setcookie(
            "selected_currency",
            $country_currency,
            time() + (2600 * 24 * 30 * 6),
            "/"
        );

        $currency = self::query()
            ->firstOrNew([
                'currency' => $country_currency
            ]);

        if ($currency->exists) {
            $_SESSION['currency'] = $currency->toArray();
            return $currency;
        }
    }
}
