<?php

namespace Mtc\Core\Providers;

use Illuminate\Support\ServiceProvider;
use \Twig_Extension_Debug;
use \Twig_SimpleTest;
use \Twig_SimpleFunction;
use \Twig_SimpleFilter;
use Mtc\Core\Currency;

/**
 * Class TwigServiceProvider
 *
 * Port for ensuring twig exists in Laravel system
 * In long term this should be replaced by a full templating engine
 * that allows adding new templates to loader
 *
 * @package Mtc\Core\Providers
 */
class TwigServiceProvider extends ServiceProvider
{
    /**
     * Register the service
     */
    public function register()
    {
        $twig_loader = new \Twig_Loader_Filesystem(\Mtc\Core\Providers\CoreModuleServiceProvider::getTemplatePaths());
        $twig_loader->addPath(dirname(__DIR__, 2) . '/templates');

        $twig_config = [
            'cache' => config('app.env') !== 'production' ? false : config('view.compiled'),
            'debug' => config('app.env') !== 'production'
        ];

        $twig = new \Twig_Environment($twig_loader, $twig_config);
        $this->app->instance('twig', $twig);
        $this->app->instance('twig_loader', $twig_loader);
        $twig->addGlobal('settings', config('settings'));
        $this->registerTwigFunctions($twig);
    }

    /**
     * Register twig extensions used by the platform
     *
     * @param \Twig_Environment $twig
     */
    public function registerTwigFunctions(\Twig_Environment $twig)
    {

        $twig->addExtension(new Twig_Extension_Debug());

        /*
         *   Extends Twig with a function that allows you to check if something is numeric
         */
        $twig->addTest(new Twig_SimpleTest('numeric', function ($value) {
            return is_numeric($value);
        }));

        /**
         * Generate the URL to a named route.
         *
         * @param  array|string  $name
         * @param  mixed  $parameters
         * @param  bool  $absolute
         * @return string
         */
        $twig->addFunction(new Twig_SimpleFunction('route', function ($name, $parameters = [], $absolute = true) {
            return route($name, $parameters, $absolute);
        }));

        /**
         * Get current request instance or obtains an input item
         *
         * @param  string  $key
         * @param  string  $default
         * @return string
         */
        $twig->addFunction(new Twig_SimpleFunction('request', function ($key = null, $default = null) {
            return request($key, $default);
        }));

        /**
         * Load config value in template
         *
         * @param  string  $key
         * @param  mixed  $default
         * @return string
         */
        $twig->addFunction(new Twig_SimpleFunction('config', function ($key, $default = null) {
            return config($key, $default);
        }));

        /**
         * Load values from previous request
         *
         * @param  string  $key
         * @param  mixed  $default
         * @return string
         */
        $twig->addFunction(new Twig_SimpleFunction('old', function ($key, $default = null) {
            return old($key, $default);
        }));

        /**
         * Get the URL of current page
         *
         * @return string
         */
        $twig->addFunction(new Twig_SimpleFunction('currentUrl', function () {
            return request()->url();
        }));

        /**
         * Pass through the url helper
         *
         * @return string
         */
        $twig->addFunction(new Twig_SimpleFunction('url', function ($path = null, $parameters = [], $secure = null) {
            return url($path, $parameters, $secure);
        }));

        /*
         *   Add twig function for returning execution time of a script.
         */
        $twig->addFunction(new Twig_SimpleFunction('execution_time', function () {
            if (DEV_MODE) {
                return round(microtime(true) - START_TIME, 4) . 's';
            }
        }));

        /*
         *   Add asset loading through versioning manifest
         */

        $twig->addFunction(new Twig_SimpleFunction('mix', function($asset_filename) {
            return config('app.asset_prefix') . mix($asset_filename);
        }));

        $twig->addFunction(new Twig_SimpleFunction('method_field', function ($method) {
            return method_field($method);
        }, [
            'is_safe' => ['html']
        ]));

        /*
         *
         *  Expose the date_create_from_format function to Twig
         *  use as {{ create_from_format($date, 'd/m/Y H:i', 'jS F Y') }}
         *
         */
        $twig->addFunction(new Twig_SimpleFunction('create_from_format', function ($date, $format_from, $format_to) {
            if (!empty($date) && !empty($format_from) && !empty($format_to)) {
                return date_create_from_format($format_from, $date)->format($format_to);
            }
        }));

        /*
         *  Generate a hash value using the contents of a given file
         */
        $twig->addFunction(new Twig_SimpleFunction('uriHash', function ($uri) {
            return $uri . '?_' . hash_file('crc32', SITE_PATH . $uri);
        }));

        /**
         * Generate the copyright range for footer
         */
        $twig->addFunction(new Twig_SimpleFunction('copyRightRange', function ($date) {
            $ret = date('Y', strtotime($date));
            if ($ret < date('Y')) {
                $ret = $ret . ' &ndash; ' . date('Y');
            }
            return $ret;
        }));

        /*
         *  Laravel translate tag
         */
        $twig->addFunction(new Twig_SimpleFunction('__', function ($key, $params = []) {
            return __($key, $params);
        }));

        /*
         *  Print_r for twig
         */
        $twig->addFunction(new Twig_SimpleFunction('d', function ($array, $format = false) {
            d($array, $format);
        }));

        /*
         *  Print_r for twig (duplicated for compatibility)
         */
        $twig->addFunction(new Twig_SimpleFunction('e', function ($array, $format = false) {
            d($array, $format);
        }));

        /*
         *
         *  Hooks function
         *  {{ action('header') }} would introduce a new action filter inside a template
         * file
         *
         */
        $twig->addFunction(new Twig_SimpleFunction('hooks_do_action', function () {
            call_user_func_array('HooksAdapter::do_action', func_get_args());
        }));

        /*
         *  Text extensions for truncate and wordwrap
         */
        $twig->addExtension(new \Twig_Extensions_Extension_Text());

        /*
         *  Date extensions for time_diff
         */

        $twig->addExtension(new \Twig_Extensions_Extension_Date());

        /*
         * Check if an array contains an array or object that has an attribute
         * with given value.
         *
         * @param $needle value of attribute
         * @param @array_of_objects
         * @param @attribute_name
         * @return bool $success
         */
        $twig->addFunction(new Twig_SimpleFunction('in_array_of_objects_or_array', function ($needle, $array_of_objects, $attribute_name = "id") {
            if (empty($array_of_objects)) {
                return false;
            }
            foreach ($array_of_objects as $object) {
                if (is_array($object)) {
                    $array = $object;
                } else {
                    $array = get_object_vars($object);
                }
                if (isset($array[$attribute_name]) && $array[$attribute_name] == $needle) {
                    return true;
                }
            }
            return false;
        }));

        /**
         * get contents of file
         *
         * @param $file string
         * @return $file string
         * @author Kieran McBurney <kieran.mcburney@mtcmedia.co.uk>
         */
        $twig->addFunction(new Twig_SimpleFunction('file_get_contents', function ($file) {

            $file_info = pathinfo($file);
            $allowed_extensions = array('svg', 'css');

            if (in_array(strtolower($file_info['extension']), $allowed_extensions)) {
                $file = file_get_contents($file);
                return $file;
            } else {
                return '';
            }
        }));

        /**
         * get the current URI
         * @return string
         * @author Aidan Ferguson <aidan.ferguson@mtcmedia.co.uk>
         */
        $twig->addFunction(new Twig_SimpleFunction('current_uri', function () {
            return request()->getRequestUri();
        }));

        /**
         * Filter user input of telephone number for output
         * @return string
         * @author Andrew Morgan <andrew.morgan@mtcmedia.co.uk>
         */
        $twig->addFunction(new Twig_SimpleFunction('filter_tel', function ($unsanitized_phone_number) {

            // Filter out all characters except numbers 0 to 9 and the plus character
            $unsanitized_phone_number = preg_replace('/[^0-9+]/', '', $unsanitized_phone_number);

            //matches country codes like +44 or 0044 at the start of a string when followed by a 0
            $regex = '/(^(\+|00)[1-9]{1,3})0/';

            //strip the predial 0 if a country code is used
            return preg_replace($regex, '$1', $unsanitized_phone_number);
        }));

        /**
         * casts values as strings and compares them if they are equal
         * @return bool
         * @author Rihards Silins <rihards.silins@mtcmedia.co.uk>
         */
        $twig->addFunction(new Twig_SimpleFunction('strings_are_equal', function ($first_value, $second_value) {
            return ((string)$first_value === (string)$second_value);
        }));

        /**
         * currency_format filter
         *
         * @param float $amount number to output
         * @param int $precision number of decimals to display
         * @param string|null $display_currency Manually specify currency to display
         * @return string the number in correct currency format
         */
        $twig->addFilter(new Twig_SimpleFilter('currency_format', function ($amount, $precision = 2, $display_currency = null) {

            if (!empty($display_currency)) {
                $selected_currency = Currency::getEnabledCurrencies()
                    ->where('currency', $display_currency)
                    ->first();

                if ($selected_currency) {
                    $currency_symbol = $selected_currency->symbol;
                    $currency_ratio = $selected_currency->ratio;
                }
            } else {
                $currency_symbol = $_SESSION['currency']['symbol'];
                $currency_ratio = $_SESSION['currency']['ratio'];

            }
            $sign = '';

            if ($amount < 0) {
                $sign = '-';
                $amount = -$amount;
            }

            return $sign . $currency_symbol . number_format($amount * $currency_ratio, $precision);
        }, [
            'is_safe' => ['html']
        ]));

        /**
         * Return enabled currencies
         *
         * @author Andrew Morgan <andrew.morgan@mtcmedia.co.uk>
         *
         * @return array Returns currency object of all enabled currencies
         */
        $twig->addFunction(new Twig_SimpleFunction('enabled_currencies', function () {
            return Currency::getEnabledCurrencies();
        }));

        /**
         * Ability to call a static method through twig
         *
         * @author Lukas Giegerich <lukas@mtcmedia.co.uk>
         * @author Stackoverflow ({@link http://stackoverflow.com/questions/6844266/twig-template-engine-using-a-static-function-or-variable})
         *
         * @param string $class
         * @param string $function
         * @param array $args
         *
         * @return mixed|null
         */

        $twig->addFunction(new Twig_SimpleFunction('staticCall', function ($class, $function, $args = []) {
            if (class_exists($class) && method_exists($class, $function)) {
                return call_user_func_array([$class, $function], $args);
            }

            return null;
        }));

        /**
         * Add a new Twig function for rendering form fields
         */
        $twig->addFunction(new Twig_SimpleFunction('form_field', function ($options) use ($twig) {
            return \Mtc\Core\FormField::renderField($options, $twig);
        }));

        /**
         * Renders image or picture fields
         *
         * @author Edvards Krickis <edvards.krickis@mtcmedia.co.uk>
         *
         * @param string $content
         *
         * @return string
         */
        $twig->addFunction(new Twig_SimpleFunction('output_image', function ($options) use ($twig) {
            return \Mtc\Core\OutputImage::renderImageField($options, $twig);
        }, [
            'is_safe' => ['html']
        ]));

        /*
         * Creates the html for all the available currencies.
         */
        $twig->addFilter( new Twig_SimpleFilter('multi_currency', function ($amount) use ($twig) {
            return $twig->render('currency/multi_currency.twig', [
                'amount' => $amount,
                'selected_currency' => $_SESSION['selected_currency']
            ]);

        }, [
            'is_safe' => ['html']
        ]));

        /**
         * Get shop search value
         */
        $twig->addFunction(new Twig_SimpleFunction('getSiteSearchValue', function () {
            return defined('SHOP_ENABLED') && SHOP_ENABLED ? getShopSearchValue() : '';
        }));

        /**
         * Check if file exists
         */
        $twig->addTest(new Twig_SimpleTest('actually_there', function ($file) {
            return file_exists($file);
        }));

        /**
         * Slugify a URL in twig
         *
         * @author Kieran McBurney <kieran.mcburney@mtcmedia.co.uk>
         *
         * @param string $url
         *
         * @return string
         */
        $twig->addFunction(new Twig_SimpleFunction('slugify', function ($url) {
            return str_slug($url);
        }));

        /**
         * Update tinymce styles for emails
         *
         * @author Kieran McBurney <kieran.mcburney@mtcmedia.co.uk>
         *
         * @param string $content
         *
         * @return string
         */
        $twig->addFunction(new Twig_SimpleFunction('email_styled_text', function ($content, $text_styles, $link_styles) {

            $content = str_replace([
                '<li>',
                '<p>',
                '<a>',
            ], [
                '<li style="' . $text_styles . '">',
                '<p style="' . $text_styles . '">',
                '<a style="' . $link_styles . '" ',
            ], $content);

            return $content;
        }));

        /**
         * Adds top padding to image as height before image loads.
         *
         * This function adds padding top in % for the temp div tag so that
         * the div block is already in need height while the image is loaded
         *
         * @author Peteris Maltenieks <peteris.maltenieks@mtcmedia.co.uk>
         * @co-author Valdis Ceirans <valdis.ceirans@mtcmedia.co.uk>
         * @co-author Edvards Krickis <edvards.krickis@mtcmedia.co.uk>
         *
         * @param string $content
         *
         * @return string
         */
        $twig->addFunction(new Twig_SimpleFunction('image_get_dimensions', function ($file, $padding = -1, $external_image = false) {
            $get_file = ($external_image) ? $file : SITE_PATH . $file;
            $file_info = pathinfo($get_file);
            $allowed_extensions = ['jpg', 'png', 'jpeg'];
            $return_data = 'data-src="' . $file . '"';

            if (!$external_image && file_exists($get_file)) {
                if (!in_array(strtolower($file_info['extension']), $allowed_extensions)) {
                    return '';
                }

                if ($padding == -1) {
                    list($width, $height) = getimagesize($get_file);
                    $ratio = $height / $width;
                    $padding = str_replace(',', '.', 100 * $ratio);
                    //              $return_data .= 'data-width="' . $width . '" data-height="' . $height . '"';  // debugging
                }
                $return_data .= ' style="padding-top: ' . $padding . '%;"';
                return $return_data;
            }

            if ($padding > 0) {
                $return_data .= ' style="padding-top: ' . $padding . '%;"';
            }

            return $return_data;
        }));
    }

}
