<?php

namespace Mtc\Core\Admin;

use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Support\Facades\Cache;
use Spatie\Permission\Traits\HasRoles;

/**
 * Admin\User class.  This looks after the admin_users table 
 * using Eloquent base modelling
 *
 * @author william.cameron
 */
class User extends Authenticatable
{
    use HasRoles;

    /**
     * Session key for checking authorization
     */
    const SESSION_KEY = 'adminId';

    /**
     * Seems to be used by moxie manager admin auth check
     */
    const ADMIN_AREA_SESSION_KEY = 'mtcadminarea';

    /**
     * No timestamps on model
     *
     * @var bool
     */
    public $timestamps = false;

    /**
     * Table name
     *
     * @var string
     */
    protected $table = "admin_users";

    /**
     * Mass assignable fields
     *
     * @var array
     */
    protected $fillable = [
        'name',
        'email',
        'username',
        'password'
    ];

    /**
     * Cast variables to specific types
     *
     * @var array
     */
    protected $casts = [
        'last_login' => 'datetime'
    ];

    /**
     * Boot model
     */
    protected static function boot()
    {
        parent::boot();

        self::saved(function ($user) {
            Cache::forget("admin-menu-{$user->id}");
        });
    }

    /**
     * Check if user has access to view this url by checking admin menu entries
     *
     * @param $url_path
     * @return bool
     */
    public function canAccessUrl($url_path)
    {
        $menu_entries = Menu::query()
            ->where('activePath', '!=', '')
            ->get()
            ->filter(function ($menu) use ($url_path) {
                return strpos($url_path, $menu->activePath) !== false;
            })
            ->pluck('permission_name');

        foreach ($menu_entries as $menu_permission) {
            if (!$this->hasPermissionTo($menu_permission)) {
                return false;
            }
        }
        return true;
    }

}
