<?php
namespace Mtc\Core\Http\Controllers\Auth;

use Illuminate\Auth\Passwords\DatabaseTokenRepository;
use Illuminate\Foundation\Application;
use Illuminate\Foundation\Auth\SendsPasswordResetEmails;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\URL;
use Illuminate\Support\Str;
use Mtc\Core\Admin\User;
use Mtc\Core\Http\Controllers\Controller;

/**
 * Class LoginController
 *
 * @package Mtc\Core\Http\Controllers\Auth
 */
class ForgotPasswordController extends Controller
{
    /*
    |--------------------------------------------------------------------------
    | Password Reset Controller
    |--------------------------------------------------------------------------
    |
    | This controller is responsible for handling password reset emails and
    | includes a trait which assists in sending these notifications from
    | your application to your users. Feel free to explore this trait.
    |
    */

    use SendsPasswordResetEmails;

    /**
     * @var Application
     */
    protected $app;

    /**
     * Render the password reset form
     *
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function show()
    {
        return template('admin/auth/forgot_password.twig');
    }

    /**
     * Handle password reset
     *
     * @param Request $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function send(Request $request)
    {
        $this->validateEmail($request);
        $this->sendResetLinkEmail($request);

        // Give a loose message to avoid leaking admin user emails
        session()->flash('success', 'An email will be sent to the given email address if it is part of the user list.');
        return redirect()->back();
    }

    /**
     * Send a reset link to the given user.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return void
     */
    public function sendResetLinkEmail(Request $request)
    {
        /** @var User $user */
        $user = User::query()->where('email', $request->input('email'))->first();
        if (!$user) {
            return;
        }

        $this->app = app();
        $config = $this->app['config']["auth.passwords.users"];
        $token_repository = $this->createTokenRepository($config);
        $token = $token_repository->create($user);

        $this->sendEmail($user, $token);
    }

    /**
     * Create token repository
     *
     * @param array $config
     * @return DatabaseTokenRepository
     */
    protected function createTokenRepository(array $config)
    {
        $key = $this->app['config']['app.key'];

        if (Str::startsWith($key, 'base64:')) {
            $key = base64_decode(substr($key, 7));
        }

        $connection = $config['connection'] ?? null;

        return new DatabaseTokenRepository(
            $this->app['db']->connection($connection),
            $this->app['hash'],
            $config['table'],
            $key,
            $config['expire']
        );
    }

    /**
     * Send the email to user
     *
     * @param User $user
     * @param $token
     */
    protected function sendEmail(User $user, $token)
    {
        $subject = 'Password Reset on ' . config('app.name');
        $content = template('emails/admin/password_reset.twig', [
            'url' => URL::signedRoute('password.show_reset', [
                'token' => $token,
                'email' => $user->email,
            ])
        ]);
        email($user->email, $subject, $content);
    }
}