<?php

namespace Mtc\Core\Admin;

use Illuminate\Database\Eloquent\Model as Eloquent;
use Illuminate\Support\Facades\File;

/**
 * Setting Class
 *
 * Legacy support for database managed settings.
 * Install and loading flow to avoid plugins breaking
 *
 * @package Mtc\Core
 * @author Craig McCreath <craig@mtcmedia.co.uk>
 * @contributor Martins Fridenbergs <martins@mtcmedia.co.uk>
 */
class Setting extends Eloquent
{
    protected static function boot()
    {
        parent::boot();

        self::creating(function (self $setting) {
           $setting->addConfig($setting->toArray());
           return false;
        });
    }

    /**
     * Legacy support for defining settings as constants
     * to avoid plugins trying to install themselves all the time
     */
    public static function defineUserSettings()
    {
        collect(config('settings', []))
            ->each(function ($value, $key) {
                if (!defined($key)) {
                    define($key, $value);
                }
            });
    }

    /**
     * Insert batch of settings into config
     *
     * Using Uppercase Insert here as it is the convention
     * used in the function call.
     *
     * @param array $data
     */
    public static function Insert($data = [])
    {
        collect($data)
            ->each(function ($setting) {
                (new self)->addConfig($setting);
            });
    }

    /**
     * Add a config to settings file
     *
     * @param $setting
     */
    public function addConfig($setting)
    {
        if (config('settings.' . $setting['key']) !== null) {
            return;
        }

        $setting['value'] = $this->convertToType($setting);
        if (!empty($setting['description'])) {
            $setting['description'] = $this->sanitiseDescription($setting);
        }

        $config_line = template('admin/config_line_builder.twig', [
            'config_line' => $setting
        ]);
        $this->pushConfigToFile($config_line);
    }

    /**
     * Convert setting value to its intended export type
     *
     * @param $setting
     * @return string
     */
    protected function convertToType($setting)
    {
        switch ($setting['type']) {
            case 'bool':
                return $setting['value'] ?  'true' : 'false';
                break;

            case 'string':
                return "'{$setting['value']}'";
                break;

            default:
                return $setting['value'];
                break;
        }
    }

    /**
     * Convert setting value to its intended export type
     *
     * @param $setting
     * @return string
     */
    protected function sanitiseDescription($setting)
    {
        // remove /**  */ blocks
        $description = str_replace(['/**', '*/'], '', $setting['description']);

        // trim multi-line comments
        $description = ltrim($description, ' *');

        return str_replace("\n", "\n     *", $description);
    }

    /**
     * Write config line to settings file
     *
     * @param $config_line
     */
    protected function pushConfigToFile($config_line)
    {
        $file_data = File::get(config_path('settings.php'));
        $updated_settings = str_replace('];', $config_line, $file_data);
        File::put(config_path('settings.php'), $updated_settings);
    }

    /**
     * Load information from installer config
     *
     * @param string $value
     * @return array|mixed|string
     */
    public static function loadInstallerConfig($value = '')
    {
        try {
            $config_file = storage_path('app/installer_config.json');
            $config = json_decode(File::get($config_file), true);

            // If we don't have a value passed through send the whole config set
            return !empty($value) ? $config[$value] : $config;
        } catch (\Exception $exception) {
            // Return empty value if config cannot be found
            return !empty($value) ? '' : [] ;
        }
    }
}