<?php

namespace Mtc\Core\Http\Controllers\Admin;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\File;
use Mtc\Core\Http\Controllers\Controller;
use Mtc\Installer\InstallResponse;
use Mtc\Installer\Facades\Installer;

/**
 * Manages the installation of modules and plugins within the Core admin.
 */
class ComponentController extends Controller
{
    /**
     * Generates the listing of plugins/components and installed ones too
     *
     * @return mixed
     */
    public function index()
    {
        $this->page_meta['title'] = 'Component Manager';
        $all_components = [
            'plugins' => Installer::availableComponents('plugins'),
            'modules' => Installer::availableComponents('modules'),
        ];

        // Loop through all components and return only installed.
        $installed_components = collect($all_components)->map(function ($components) {
            return $components->filter(function ($component) {
                return $component['installed'] === true;
            });
        });

        return template('admin/components/index.twig', [
            'page_meta' => $this->page_meta,
            'all_components' => $all_components,
            'installed_components' => $installed_components,
        ]);
    }


    /**
     * Validates the data before looping through each of the plugins/components and
     * installing those which were checked in the form
     *
     * @param Request $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function store(Request $request)
    {
        $this->validate($request, [
            'bitbucket_username' => 'required',
            'bitbucket_password' => 'required',
            'install_component' => 'required|array',
        ]);

        Cache::forget('install-config');
        Cache::remember('install-config', now()->addMinutes(10), function () use ($request) {
            return [
                'bb_username' => $request->input('bitbucket_username'),
                'bb_password' => $request->input('bitbucket_password'),
            ];
        });

        $all_components = [
            'plugins' => Installer::availableComponents('plugins'),
            'modules' => Installer::availableComponents('modules'),
        ];

        $installed_directly = collect($all_components)
            ->map(function ($group, $group_name) use ($request) {
                return $group
                    ->filter(function ($component) use ($group_name, $request) {
                        return in_array($component['name'], $request->input("install_component.{$group_name}", []));
                    })
                    ->map(function($component_to_install) {
                        $install_path = str_replace('core/', '', $component_to_install['name']);
                        return Installer::installPlugin($component_to_install['url'], $install_path);
                    });
            })
            ->flatten(1)
            ->reject(function (InstallResponse $component) {
                return $component->failed();
            });

        $second_level_dependencies = $installed_directly
            ->map(function (InstallResponse $component) {
                return $component->dependencies()
                    ->reject(function ($dependency_url, $path) {
                        return File::exists(base_path($path));
                    });
            })
            ->flatten(1)
            ->map(function($component_to_install) {
                $install_path = str_replace('core/', '', $component_to_install['name']);
                return Installer::installPlugin($component_to_install['url'], $install_path);
            })
            ->reject(function (InstallResponse $component) {
                return $component->failed();
            });

        $installed_count = $installed_directly->count() + $second_level_dependencies->count();

        // Set message and redirect
        if ($installed_count == 0) {
            session()->flash('error', 'Please check your bitbucket username/password.');
        } else {
            Cache::forget("installer-components-plugins");
            Cache::forget("installer-components-modules");
            session()->flash('success', $installed_count . ' components installed.');
        }

        return redirect()->back();
    }
}
