<?php

namespace Mtc\Core\Http\Controllers\Admin;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Mtc\Core\Admin\Role;
use Mtc\Core\Http\Controllers\Controller;
use Mtc\Core\Admin\User;

/**
 * Class AdminUsersController
 */
class AdminUsersController extends Controller
{
    /**
     * Admin Users page
     *
     * @return string
     * @throws \Twig_Error_Loader
     * @throws \Twig_Error_Runtime
     * @throws \Twig_Error_Syntax
     */
    public function index()
    {
        $this->page_meta['title'] = 'Manage Admin Users';

        $users = User::query();

        if (!$this->admin_user->hasRole('mtc')) {
            $users->whereDoesntHave('roles', function ($role_query) {
                return $role_query->where('name', 'mtc');
            });
        }
        $users = $users->latest('id')
            ->get();

        $twig_params = [
            'page_meta' => $this->page_meta,
            'admin_users' => $users,
        ];

        return template('admin/users/index.twig', $twig_params);
    }

    /**
     * Create Page
     *
     * @return string
     * @throws \Twig_Error_Loader
     * @throws \Twig_Error_Runtime
     * @throws \Twig_Error_Syntax
     */
    public function create()
    {
        $this->page_meta['title'] = 'Create Admin User Entry';

        $roles = Role::query()
            ->pluck('name', 'id')
            ->reject(function ($role) {
                if (!$this->admin_user->hasRole('mtc')) {
                    return $role === 'mtc';
                }
            });

        $twig_params = [
            'page_meta' => $this->page_meta,
            'user' => new User(),
            'current_user' => $this->admin_user,
            'roles' => $roles,
        ];

        return template('admin/users/edit.twig', $twig_params);
    }

    /**
     * Parse create entry request
     *
     * @param Request $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function store(Request $request)
    {
        $validated = $this->validate($request, [
            'name' => 'required',
            'email' => 'required|email|unique:admin_users,email',
            'password' => 'required|min:6|confirmed',
            'roles' => 'required',
        ]);

        $user = User::query()
            ->create([
                'password' => bcrypt($request->input('password')),
                'email' => $request->input('email'),
                'name' => $request->input('name')
            ]);

        $user->roles()->sync($validated['roles']);

        session()->flash('success', 'Admin user created');
        return redirect("/admin/users/{$user->id}/edit");
    }

    /**
     * Edit Page
     *
     * @return string
     * @throws \Twig_Error_Loader
     * @throws \Twig_Error_Runtime
     * @throws \Twig_Error_Syntax
     */
    public function edit($admin_user_id)
    {
        $user = User::query()->find($admin_user_id);
        if ($user->hasRole('mtc') && !$this->admin_user->hasRole('mtc')) {
            session()->flash('error', 'You cannot edit this account.');
            return redirect('/admin/users');
        }

        $this->page_meta['title'] = 'Update Admin User Entry';


        $roles = Role::query()
            ->pluck('name', 'id')
            ->reject(function ($role) {
                if (!$this->admin_user->hasRole('mtc')) {
                    return $role === 'mtc';
                }
            });

        $twig_params = [
            'page_meta' => $this->page_meta,
            'user_data' => $user,
            'user' => $user,
            'current_user' => $this->admin_user,
            'roles' => $roles,
        ];

        return template('admin/users/edit.twig', $twig_params);
    }

    /**
     * Parse update request
     *
     * @param Request $request
     * @param $admin_user_id
     * @return \Illuminate\Http\RedirectResponse
     */
    public function update(Request $request, $admin_user_id)
    {
        $rules = [
            'name' => 'required',
            'roles' => 'required',
            'email' => 'required|email|unique:admin_users,email,' . $admin_user_id,
        ];

        if ($request->input('password')) {
            $rules['password'] = 'required|min:6|confirmed';
        }

        $validated = $this->validate($request, $rules);

        /** @var User $user */
        $user = User::query()->find($admin_user_id);
        if ($request->input('password')) {
            $validated['password'] = bcrypt($request->input('password'));
        }

        $user->fill($validated);
        $user->save();

        $user->roles()->sync($validated['roles']);

        session()->flash('success', 'Admin user updated');
        return redirect()->back();
    }

    /**
     * Delete admin user entry
     *
     * @param int $admin_user_id
     */
    public function destroy($admin_user_id)
    {
        $user = User::query()->findOrFail($admin_user_id);
        if ($user->hasRole('mtc') || $user->id == $this->admin_user->id) {
            session()->flash('error', 'You cannot delete this account.');
            return redirect()->back();
        }

        $user->delete();
        session()->flash('success', 'Admin User Removed');
        return redirect('/admin/users');
    }

    /**
     * Login As different admin user
     *
     * @param int $admin_user_id
     */
    public function login($admin_user_id)
    {
        $user = User::query()->find($admin_user_id);
        Auth::login($user);
        session()->put(User::SESSION_KEY, $user->id);
        return redirect('/admin/');
    }

}
