<?php

namespace Mtc\Core\Images;

use Symfony\Component\Finder\Finder;
/**
 * Image Optimise class
 *
 * Losslessly optimise image (optimized file overwrites original one).
 * Depends on https://packagist.org/packages/ps/image-optimizer
 *
 * @author Reinis Liepkalns | mtc.
 * @author Paul McAvoy | mtc.
 * @version 2018-07-01
*/
class ImageOptimise
{
    /**
    * optimizeLosslessly method
    *
    * This method will attempt to losslessly optimise an image. If it
    * succeeds, it will then overwrite the original one with the new optimised.
    *
    * @param string $filepath
    * @param boolean $enabled
    */
    public static function optimizeLosslessly($filepath, $enabled = IMAGE_OPTIMISE_LOSSLESS)
    {
        if ($enabled === true) {
            $factory = new \ImageOptimizer\OptimizerFactory([
                'jpegtran_bin' => '/usr/local/bin/jpegtran',
                'jpegoptim_bin' => '/usr/local/bin/jpegoptim',
                'pngquant_bin' => '/usr/local/bin/pngquant',
                'optipng_bin' => '/usr/local/bin/optipng',
                'pngcrush_bin' => '/usr/local/bin/pngcrush',
                'advpng_bin' => '/usr/local/bin/advpng'
            ]);

            $optimizer = $factory->get();

            //optimized file overwrites original one
            $optimizer->optimize($filepath);
        }
    }

    /**
     * Runs imagemagick's convert command to strip EXIF and reduce quality to 85%
     *
     * @param mixed $filepath
     * @param int $quality
     * @return mixed $result
     */
    public static function optimizeLossy($filepath, $quality = 80)
    {
        $command = IMAGE_CONVERT . " {$filepath} -auto-orient -strip -quality {$quality} {$filepath}";
        return shell_exec($command);
    }

    /**
     * Function that get images passed to it and optimises them
     *
     * @param string $image_path
     * @param int $quality
     */
    public static function optimiseImage($image_path, $quality)
    {
        // get $image extension
        $extension = pathinfo($image_path, PATHINFO_EXTENSION);
        // if image is jpg the
        if (in_array(strtolower($extension), ['jpg', 'jpeg'])) {
            self::optimizeLossy($image_path, $quality);
        }
        // run no destructive optimisation
        self::optimizeLosslessly($image_path, IMAGE_OPTIMISE_LOSSLESS);
    }

    /**
     * Function that get images passed to it and optimises them
     *
     * @param array $folders
     * @param string $file_age
     * @param bool $print
     * @param int $quality pass a value to destructively optimise images
     */
    public static function bulkOptimiseImages(array $folders, $file_age, $quality = 100, $print = true)
    {
        $finder = new Finder();
        // iterate over folders passed to function
        foreach ($folders as $folder) {
            // add SITE_PATH to the folder path
            $directory = base_path() . DIRECTORY_SEPARATOR . trim($folder, DIRECTORY_SEPARATOR);

            // check that the directory exists
            if (!file_exists($directory)) {
                continue;
            }

            // use symfony to grab all files recursively within $directory
            $images = $finder->files()->in($directory);

            if (!empty($file_age)) {
                $images->date($file_age);
            }

            // iterate the files in the originals folder and simulate an image upload
            foreach ($images as $image) {
                // get $image path
                $image_path = $image->getRealPath();
                // get $image extension
                $extension = pathinfo($image_path, PATHINFO_EXTENSION);
                // if image is jpg the
                if (in_array(strtolower($extension), ['jpg', 'jpeg'])) {
                    self::optimizeLossy($image_path, $quality);
                    // print if destructive optimisation
                    if ($print && $quality < 100) {
                        echo 'Lossy optimise ' . $image->getFilename() . ' quality ' . $quality . '<br>';
                    }
                }
                // run no destructive optimisation
                self::optimizeLosslessly($image_path, IMAGE_OPTIMISE_LOSSLESS);
                if ($print) {
                    echo 'Losslessly optimise ' . $image->getFilename() . '<br>';
                }
            }
        }
    }
}
