<?php

namespace Mtc\Core\Providers;

use Illuminate\Support\Facades\File;
use Illuminate\Support\ServiceProvider;
use Mtc\Cms\Providers\CmsServiceProvider;
use Mtc\Shop\Providers\ShopServiceProvider;
use Mtc\Abs\Providers\AbsServiceProvider;

/**
 * Class CoreServiceProvider
 *
 * @package Mtc\Core\Providers
 */
class CoreModuleServiceProvider extends ServiceProvider
{
    /**
     * List of supported root level modules
     *
     * @var array
     */
    public static $root_level_module_service_providers = [
        CmsServiceProvider::class,
        ShopServiceProvider::class,
        AbsServiceProvider::class,
    ];

    /**
     * Get the hook files for components
     *
     * @return \Illuminate\Support\Collection
     */
    public static function getComponentHooks()
    {
        return collect(app('installed_components'))
            ->map(function ($component) {
                return $component . '/hooks.php';
            })
            ->filter(function ($component) {
                return File::exists($component);
            });
    }

    /**
     * Get the loader files for components
     *
     * @return \Illuminate\Support\Collection
     */
    public static function getComponentLoaders()
    {
        return collect(app('installed_components'))
            ->map(function ($component) {
                return $component . '/loader.php';
            })
            ->filter(function ($component) {
                return File::exists($component);
            });
    }

    /**
     * Get the template paths
     *
     * @return array
     */
    public static function getTemplatePaths()
    {
        $template_paths = collect([
            resource_path('templates/'),
            base_path('sites/' . config('app.site_id') . '/templates/'),
            base_path('core/css/'),
        ]);

        $component_paths = collect(app('installed_components'))
            ->map(function ($component) {
                return $component . '/templates/';
            })
            ->filter(function ($component) {
                return File::exists($component) && File::isDirectory($component);
            });

        $template_paths = $template_paths->merge($component_paths);

        if (File::exists(base_path('shop/templates'))) {
            $template_paths->push(base_path('shop/templates/'));
        }

        if (File::exists(base_path('cms/templates'))) {
            $template_paths->push(base_path('cms/templates/'));
        }

        if (File::exists(base_path('abs/templates'))) {
            $template_paths->push(base_path('abs/templates/'));
        }

        return $template_paths
            ->filter(function ($path) {
                return File::exists($path);
            })->toArray();
    }

    /**
     * Register any application services.
     */
    public function register()
    {
        collect(self::$root_level_module_service_providers)
            ->filter(function ($service_provider_class) {
                return class_exists($service_provider_class);
            })
            ->each(function ($service_provider_class) {
                $this->app->register($service_provider_class);
            });

        $this->loadComponents();
    }

    /**
     * Load list of installed components
     */
    protected function loadComponents()
    {
        $components = [];
        foreach (config('core.component_install_paths', []) as $type) {

            // Skip if this component type dir doesn't exist
            if (!File::isDirectory(base_path($type . '/'))) {
                continue;
            }

            $type_components = $this->getOrderedComponentListForType($type);
            $components = array_merge($components, $type_components);
            $this->tryRegisteringComponents($type_components, $type);
        }
        $this->app->instance('installed_components', $components);
    }

    /**
     * Attempt registering components
     * Will be done if they have an existing service provider class
     *
     * @param $components
     * @param $type
     */
    protected function tryRegisteringComponents($components, $type)
    {
        collect($components)
            ->each(function ($component) use ($type) {
                $provider_class_name = "Mtc\\" . ucfirst($type) . "\\" . basename($component)
                    ."\\Classes\\Providers\\" . basename($component) . "ServiceProvider";

                if (class_exists($provider_class_name)) {
                    $this->app->register($provider_class_name);
                }
            });
    }

    /**
     * Get the list of components in this section
     *
     * @param $type
     * @return \Illuminate\Support\Collection
     */
    protected function getOrderedComponentListForType($type)
    {
        $modules = collect([]);

        // check if there is a user-defined module order
        $order_file = base_path("{$type}/.order");
        if (File::exists($order_file)) {
            $modules = collect(explode(PHP_EOL, File::get($order_file)))
                ->map(function ($module) {
                    return trim($module);
                })
                ->filter();
        }

        // add unlisted modules from the modules directory
        return $modules->merge(glob(base_path("{$type}/*"), GLOB_ONLYDIR))
            ->unique()
            ->toArray();
    }
}
