<?php

namespace Mtc\Core\Admin;

use \Illuminate\Database\Eloquent\Model;
use CMSNav;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Str;
use Spatie\Permission\Models\Permission;
use Spatie\Permission\Models\Role;

/**
 * Admin\Menu class.  This looks after the admin_menu table
 * using Eloquent base modelling
 *
 * @author william.cameron
 */
class Menu extends Model
{
    /**
     * Mass assignable fields
     *
     * @var string[] $fillable
     */
    protected $fillable = [
        'title',
        'sub_id',
        'icon',
        'path',
        'activePath',
        'new_window',
        'constant',
        'order',
    ];

    /**
     * Model does not have timestamps
     *
     * @var bool
     */
    public $timestamps = false;

    /**
     * Table name
     *
     * @var string
     */
    protected $table = "admin_menu";

    /**
     * Extend model booting
     */
    protected static function boot()
    {
        parent::boot();

        self::creating(function (self $menu) {
            $menu->permission_name = Str::slug($menu->title);
        });

        self::created(function (self $menu) {
            if ($menu->sub_id > 0) {
                $permission = Permission::query()
                    ->create([
                        'name' => $menu->permission_name,
                        'guard_name' => 'web',
                        'group' => $menu->parent->title ?? '',
                        'model_group' => 'admin',
                    ]);

                $mtc_admin = Role::query()
                    ->where('group', 'admin')
                    ->where('name', 'mtc')
                    ->first();

                if ($mtc_admin) {
                    $mtc_admin->givePermissionTo($permission->name);
                }
            }
        });

        self::saved(function ($menu) {
            self::invalidateMenuCache();
        });

        self::deleted(function ($menu) {
            self::invalidateMenuCache();
        });

        self::addGlobalScope(function ($query) {
            return $query->orderBy('order', 'asc');
        });
    }

    /**
     * Does this menu item have sub items
     *
     * @return bool
     */
    public function hasSubItems()
    {
        return ! $this->subItems->isEmpty();
    }

    /**
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function children()
    {
        return $this->hasMany(self::class, 'sub_id');
    }

    /**
     * configures and handles the relation of subitems for this menu item
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function subItems()
    {
        return $this->hasMany(self::class, 'sub_id', 'id')
            ->orderBy('order', 'ASC')
            ->orderBy('title', 'ASC');
    }

    /**
     * Define relationship with parent menu item
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     * @author Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
     */
    public function parent()
    {
        return $this->belongsTo(self::class, 'sub_id');
    }

    /**
     * Scope - ChildOf()
     * Define scope to check if menu item is a child of different menu item
     *
     * @param \Illuminate\Database\Eloquent\Builder $query Query builder object
     * @param int $menu_id parent menu item ID
     * @return \Illuminate\Database\Eloquent\Builder
     * @author Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
     */
    public function scopeChildOf($query, $menu_id)
    {
        return $query->where('sub_id', $menu_id);
    }

    /**
     * Scope - isRoot()
     * Find only root level menu item
     *
     * @param \Illuminate\Database\Eloquent\Builder $query Query builder object
     * @return \Illuminate\Database\Eloquent\Builder
     * @author Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
     */
    public function scopeIsRoot($query)
    {
        return $query->ChildOf(0);
    }

    /**
     * Check whether the current admin user is allowed to view this page
     *
     * @param
     * @return bool whether user is allowed to this admin menu
     */
    public function allowed($admin_user)
    {

        if ($this->hasEnabledConstant()) {
            return true;
        }

        if (empty($this->permission_name)) {
            return true;
        }

        if ($admin_user instanceof User) {
            return $admin_user->hasPermissionTo($this->permission_name);
        }

        return false;
    }

    /**
     * Check if the menu entry setting is disabled for user
     *
     * @return bool
     */
    protected function hasEnabledConstant()
    {
        if (empty($this->constant)) {
            return false;
        }

        if (Config::has($this->constant)) {
            return Config::get($this->constant) === true;
        }


        return !(defined($this->constant) && constant($this->constant));
    }

    /**
     * Check whether menu item has children that are active
     * @return bool whether menu item has children that are active
     * @author Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
     */
    public function hasActiveChild()
    {
        $is_child_active = false;
        foreach ($this->subItems as $child_menu) {
            if (strstr($_SERVER['SCRIPT_NAME'], $child_menu->activePath)) {
                $is_child_active = true;
            }
        }
        return $is_child_active;
    }

    /**
     * Check whether menu item is in active path
     * @return bool whether the menu item is within active path
     * @author Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
     */
    public function isActive()
    {
        return stripos(request()->getRequestUri(), $this->activePath) !== false;
    }

    /**
     * Check if CMS tree should be shown for this menu item
     * @return string CMS tree under this item or empty string
     * @author Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
     * //TODO: Move this to CMS
     */
    public static function showCMSTree(User $admin_user)
    {
        if (stripos(request()->getRequestUri(), 'admin/content_manager') === false) {
            return '';
        }

        $site_tree = CMSNav::siteTree(0, PHP_INT_MAX, [
            'check_innav' => false,
            'check_published' => false
        ]);


        $is_admin = $admin_user->hasRole('mtc');

        $options = [
            'first_container_id' => 'site-tree',
            'title_container_tag' => 'span',
            'active_page_id' => request()->input('page', 0),
            'is_admin' => $is_admin
        ];

        ob_start();
        CMSNav::generateSiteTreeMenu($site_tree, $options);
        return ob_get_clean();
    }

    /**
     * Get the root level tree.
     * Used for admin menu management to allow choosing parent items
     *
     * @param int $exclude_id
     * @param bool $attach_root
     * @return \Illuminate\Database\Eloquent\Builder[]|\Illuminate\Database\Eloquent\Collection|\Illuminate\Support\Collection
     */
    public static function getRootLevelTree($exclude_id = 0, $attach_root = false)
    {
        $tree = Menu::query()
            ->where('id', '!=', $exclude_id)
            ->where('sub_id', 0)
            ->get()
            ->keyBy('id')
            ->map(function ($menu_item) {
                return $menu_item->title;
            });

        if ($attach_root) {
            $tree = $tree->prepend('Root', 0);
        }
        return $tree;
    }

    /**
     * Build the admin menu tree
     *
     * @param $admin_user
     * @return mixed
     */
    public static function generateMenuTree($admin_user)
    {
        return Cache::rememberForever("admin-menu-{$admin_user->id}", function () use ($admin_user) {
            return self::with('subItems')
                ->orderBy('order', 'ASC')
                ->isRoot()
                ->get()
                ->each(function (self $menu_item) use ($admin_user) {
                    $menu_item->sub_items = $menu_item->subItems
                        ->filter(function (self $sub_item) use ($admin_user) {
                            return $sub_item->allowed($admin_user);
                        })
                        ->each(function (self $sub_item) {
                            $sub_item->setHidden([
                                'permission_name',
                                'constant',
                                'created_at',
                                'updated_at'
                            ]);
                        })
                        ->values();

                    // Set the child relation as hidden as we are mapping them here
                    $menu_item->setHidden([
                        'permission_name',
                        'constant',
                        'subItems',
                        'created_at',
                        'updated_at'
                    ]);
                })
                ->reject(function (self $menu_item) use ($admin_user) {
                    // We can't allow non-mtc users to see mtc only
                    if (!$admin_user->hasRole('mtc') && $menu_item->permission_name === 'mtc-only') {
                        return true;
                    }

                    // reject menus without children
                    if (count($menu_item->subItems) == 0) {
                        return true;
                    }


                    // Reject elements without any allowed children
                    return $menu_item->sub_items->count() == 0;
                })
                ->values();
        })
            ->each(function ($menu_item) {
                $menu_item->subItems
                    ->each(function($sub_item) {
                        $sub_item->is_active = $sub_item->isActive();
                    });
            });
    }

    /**
     * Add menu entry access to roles
     *
     * @param self $menu
     * @param $roles
     */
    public static function grantMenuPermissionToRole(self $menu, $roles)
    {
        foreach ($roles as $role_name) {
            $role = \Mtc\Core\Admin\Role::query()
                ->where('name', $role_name)
                ->first();
            if ($role) {
                $role->givePermissionTo($menu->permission_name);
            }
        }
    }

    /**
     * Invalidate admin menu cache
     */
    public static function invalidateMenuCache()
    {
        User::all()
            ->each(function ($admin_user) {
                Cache::forget("admin-menu-{$admin_user->id}");
            });
    }
}
