<?php

namespace Mtc\Core\Http\Controllers\Admin;

use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Mtc\Core\Admin\Role;
use Mtc\Core\Http\Controllers\Controller;
use Mtc\Core\Admin\Permission;

/**
 * Class RoleController
 *
 * @package Mtc\Core\Http\Controllers\Admin
 */
class RoleController extends Controller
{
    /**
     * List Roles
     *
     * @return string
     * @throws \Twig_Error_Loader
     * @throws \Twig_Error_Runtime
     * @throws \Twig_Error_Syntax
     */
    public function index()
    {
        $this->page_meta['title'] = 'Manage User Roles';
        $roles = Role::latest('id')->get();

        return template('admin/roles/index.twig', [
            'page_meta' => $this->page_meta,
            'roles' => $roles,

        ]);
    }

    /**
     * Render create view
     *
     * @return string
     * @throws \Twig_Error_Loader
     * @throws \Twig_Error_Runtime
     * @throws \Twig_Error_Syntax
     */
    public function create()
    {
        $this->page_meta['title'] = 'Create Role';
        $permission_groups = Permission::all()
            ->each(function ($permission) {
                $permission->title = Str::title(str_replace('-', ' ', $permission->name));
            })
            ->reject(function ($permission) {
                return !$this->admin_user->hasPermissionTo($permission->name);
            })
            ->groupBy('group');

        $twig_params = [
            'page_meta' => $this->page_meta,
            'role' => new Role(),
            'permission_groups' => $permission_groups
        ];

        return template('admin/roles/edit.twig', $twig_params);
    }

    /**
     * Process creation request
     *
     * @param Request $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function store(Request $request)
    {
        $validated = $this->validate($request, [
            'name' => 'required'
        ]);

        $role = Role::query()->create($validated);
        $role->permissions()->sync($request->input('permissions'));

        session()->flash('success', 'Role created successfully');
        return redirect()->to("/admin/roles/{$role->id}/edit");
    }

    /**
     * Render role edit view
     *
     * @param $role_id
     * @return string
     * @throws \Twig_Error_Loader
     * @throws \Twig_Error_Runtime
     * @throws \Twig_Error_Syntax
     */
    public function edit($role_id)
    {
        $this->page_meta['title'] = 'Edit Role';
        $permission_groups = Permission::all()
            ->each(function ($permission) {
                $permission->title = Str::title(str_replace('-', ' ', $permission->name));
            })
            ->reject(function ($permission) {
                return !$this->admin_user->hasPermissionTo($permission->name);
            })
            ->groupBy('group');

        return template('admin/roles/edit.twig', [
            'page_meta' => $this->page_meta,
            'role' => Role::query()->find($role_id),
            'permission_groups' => $permission_groups
        ]);
    }

    /**
     * Process the role edit form
     *
     * @param Request $request
     * @param $role_id
     * @return \Illuminate\Http\RedirectResponse
     */
    public function update(Request $request, $role_id)
    {
        $validated = $this->validate($request, [
            'name' => 'required'
        ]);

        $role = Role::query()->find($role_id);
        $role->fill($validated);
        $role->save();

        $role->permissions()->sync($request->input('permissions'));

        session()->flash('success', 'Role updated');
        return redirect()->back();
    }

    /**
     * Delete a role
     *
     * @param $role_id
     * @return \Illuminate\Http\RedirectResponse
     * @throws \Exception
     */
    public function destroy($role_id)
    {
        $role = Role::query()->find($role_id);
        if ($role->name === 'mtc') {
            session()->flash('error', 'You cannot delete this user role');
            return redirect()->back();
        }

        $role->delete();
        session()->flash('success', 'Role Removed');
        return redirect()->back();
    }

}
