<?php

namespace Mtc\Core\Seo;

use Illuminate\Database\Eloquent\Model;

/**
 * Seo Redirect class
 *
 * The Seo Redirect class contains method which redirects to a new URL.
 *
 * @author Aivars Francis | mtc.
 * @version 2014-01-12
 */
class Redirect extends Model
{
    /**
     * Table name
     *
     * @var string
     */
    protected $table = 'seo_redirects';

    /**
     * Disable eloquent default timestamp columns
     *
     * @var bool
     */
    public $timestamps = false;

    /**
     * Mass-assignable table attributes
     *
     * @var array
     */
    protected $fillable = [
        'code',
        'old_url',
        'new_url',
        'hits'
    ];

    /**
     * List with redirect codes
     *
     * @var array
     */
    public static $REDIRECT_CODES = array(
        '301' => 'Moved Permanently (301)',
        '302' => 'Found (302)',
        '303' => 'See Other (303)',
        '307' => 'Moved Temporarily (307)'
    );

    /**
     * Redirects page to new url
     *
     * @param string $path
     */
    public static function redirectToPage($path)
    {
        $redirect = self::query()
            ->where('old_url', '=', $path);

        // Facebook tends to add "?fbclid=sdivui7fy34bkfygkuygjsd" to the posted URLs so need to parse them
        $parsed_url = parse_url($path);
        $cleaned_path = '';
        $query_to_add = '';
        if (!empty($parsed_url)) {
            if (array_key_exists('path', $parsed_url) && !empty($parsed_url['path'])) {
                $cleaned_path = $parsed_url['path'];
            }

            if (array_key_exists('query', $parsed_url) && !empty($parsed_url['query'])) {
                $query_to_add = $parsed_url['query'];
            }
        }

        if (!empty($cleaned_path)) {
            $redirect = $redirect->orWhere('old_url', '=', $cleaned_path);
        }

        // match with some rule
        if ($redirect->count() > 0) {
            //get rule
            $redirect = $redirect->first();
            $redirect->hits = $redirect->hits + 1;
            $redirect->update();

            $new_url = $redirect->new_url;
            if (!empty($query_to_add)) {
                $new_url .= '?' . $query_to_add;
            }

            header("Location: " . $new_url, true, $redirect->code);
            exit();
        }
    }

    /**
     * Query for export
     *
     * @return \Illuminate\Database\Eloquent\Builder
     */
    public static function exportQuery()
    {
        return self::query()
            ->select([
                'code',
                'old_url',
                'new_url'
            ])
            ->orderBy('id');
    }
}
