<?php

namespace Mtc\Core\Http\Controllers\Admin;

use Illuminate\Http\Request;
use Mtc\Core\Http\Controllers\Controller;
use Mtc\Core\Seo\Redirect;

/**
 * Manages the installation of modules and plugins within the Core admin.
 */
class SeoRedirectController extends Controller
{

    /**
     * Displays the list of Redirects
     *
     * @return mixed
     */
    public function index()
    {
        $redirect_list = Redirect::query()
            ->get();

        // List the Error Codes
        $redirect_codes = Redirect::$REDIRECT_CODES;

        $this->page_meta['title'] = 'Redirects';
        return template('admin/seo/redirects/index.twig', [
            'page_meta' => $this->page_meta,
            'redirect_codes' => $redirect_codes,
            'redirect_list' => $redirect_list,
        ]);
    }

    /**
     *  Validates the data from form and inserts into Redirects table
     *
     * @param Request $request
     *
     * @return \Illuminate\Http\RedirectResponse
     */
    public function store(Request $request)
    {
        $this->validate($request, [
            'old_url' => ['required', 'regex: #^/#'],
            'new_url' => ['required', 'regex: #^/#'],
            'code' => 'required',
        ]);

        $redirect = Redirect::firstOrCreate([
            'old_url' => $request->input('old_url'),
            'new_url' => $request->input('new_url'),
            'code' => $request->input('code'),
        ]);
        $redirect->save();

        session()->flash('success', 'Your redirect method has now been added!');
        return redirect()->back();
    }

    /**
     * Validates the data from the form and updates the Redirects table
     *
     * @param Request $request
     * @param int $redirect_id
     *
     * @return \Illuminate\Http\RedirectResponse
     */
    public function update(Request $request, $redirect_id)
    {
        $this->validate($request, [
            'old_url' => ['required', 'regex: #^/#'],
            'new_url' => ['required', 'regex: #^/#'],
            'code' => 'required'
        ]);

        Redirect::where('id', $redirect_id)
            ->update([
                'old_url' => $request->input('old_url'),
                'new_url' => $request->input('new_url'),
                'code' => $request->input('code'),
            ]);

        session()->flash('success', 'Your redirect method has now been updated!');
        return redirect()->back();
    }

    /**
     * Displays form filled out with data depending on ID
     *
     * @param int $redirect_id
     *
     * @return mixed
     */
    public function edit($redirect_id)
    {
        $this->page_meta['title'] = 'Redirects';
        $redirect_codes = Redirect::$REDIRECT_CODES;

        $redirects = Redirect::where('id', $redirect_id)->get();

        return template('admin/seo/redirects/edit.twig', [
            'page_meta' => $this->page_meta,
            'redirects' => $redirects,
            'redirect_codes' => $redirect_codes,
        ]);
    }

    /**
     * Deletes the Redirect depending on the ID
     *
     * @param int $redirect_id
     *
     * @return \Illuminate\Http\RedirectResponse
     */
    public function destroy($redirect_id)
    {
        $delete_canonical = Redirect::find($redirect_id);
        $delete_canonical->delete();

        session()->flash('success', 'Your redirect has now been deleted!');
        return redirect()->back();
    }
}
