<?php

namespace Mtc\Core\Http\Controllers\Admin;

use Illuminate\Http\Request;
use Mtc\Core\Admin\Menu;
use Mtc\Core\Admin\Role;
use Mtc\Core\Http\Controllers\Controller;

/**
 * Class MenuController
 */
class MenuController extends Controller
{
    /**
     * Listing page
     *
     * @return string
     * @throws \Twig_Error_Loader
     * @throws \Twig_Error_Runtime
     * @throws \Twig_Error_Syntax
     */
    public function index()
    {
        $menu_tree = Menu::query()
            ->isRoot()
            ->with('children.children')
            ->get();

        $this->page_meta['title'] = 'Manage Admin Menu';
        return template('admin/menu/index.twig', [
            'menu_tree' => $menu_tree,
            'page_meta' => $this->page_meta,
        ]);
    }

    /**
     * Create Page
     *
     * @return string
     * @throws \Twig_Error_Loader
     * @throws \Twig_Error_Runtime
     * @throws \Twig_Error_Syntax
     */
    public function create()
    {
        $this->page_meta['title'] = 'Create Admin Menu Entry';
        return template('admin/menu/edit.twig', [
            'menu_item' => new Menu(),
            'menu_options' => Menu::getRootLevelTree(0, true),
            'page_meta' => $this->page_meta,
        ]);
    }

    /**
     * Parse create entry request
     *
     * @param Request $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function store(Request $request)
    {
        $this->validate($request, [
            'title' => 'required',
            'path' => 'required_unless:sub_id,0'
        ]);

        $menu_item = new Menu();
        $menu_item->fill($request->input());
        $menu_item->new_window = $request->has('new_window') ? 1 : 0;
        $menu_item->save();

        if ($request->has('grant_access')) {
            Role::all()
                ->each(function (Role $role) use ($menu_item) {
                    $role->givePermissionTo($menu_item->permission_name);
                });
        }

        session()->flash('success', 'Menu Item Added');
        return redirect()->to("/admin/menu/{$menu_item->id}/edit");
    }

    /**
     * Edit Page
     *
     * @return string
     * @throws \Twig_Error_Loader
     * @throws \Twig_Error_Runtime
     * @throws \Twig_Error_Syntax
     */
    public function edit($menu_id)
    {
        $this->page_meta['title'] = 'Update Admin Menu Entry';
        $menu_item = Menu::query()->findOrFail($menu_id);

        return template('admin/menu/edit.twig', [
            'menu_item' => $menu_item,
            'menu_options' => Menu::getRootLevelTree($menu_id, true),
            'page_meta' => $this->page_meta,
        ]);
    }

    /**
     * Parse update request
     *
     * @param Request $request
     * @param $menu_id
     * @return \Illuminate\Http\RedirectResponse
     */
    public function update(Request $request, $menu_id)
    {
        $this->validate($request, [
            'title' => 'required',
            'path' => 'required_unless:sub_id,0'
        ]);

        $menu_item = Menu::query()->findOrFail($menu_id);
        $menu_item->fill($request->input());
        $menu_item->new_window = $request->has('new_window') ? 1 : 0;
        $menu_item->save();

        session()->flash('success', 'Menu Item Updated');
        return redirect()->back();
    }

    /**
     * Delete menu entry
     *
     * @param int $menu_id
     */
    public function destroy($menu_id)
    {
        $menu = Menu::query()->find($menu_id);
        if ($menu) {
            $menu->delete();
            session()->flash('success', 'Menu Item Removed');
        }
        return redirect()->back();
    }

    /**
     * Update menu order
     *
     * @param Request $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function reorder(Request $request)
    {
        parse_str($request->input('values'), $page_order);
        collect($page_order['page'])
            ->mapToGroups(function ($parent, $page_id) {
                if ($parent === 'null') {
                    $parent = 0;
                }
                return [$parent => $page_id];
            })
            ->each(function ($group, $parent) {
                $group->each(function ($page_id, $order) use ($parent) {
                    Menu::query()
                        ->where('id', $page_id)
                        ->update([
                            'sub_id' => $parent,
                            'order' => $order
                        ]);
                });
            });

        session()->flash('success', 'Menu Item Order Updated');
        return redirect()->back();
    }
}
