<?php

namespace Mtc\Core\Seo;

use Illuminate\Http\Request;
use Illuminate\Http\UploadedFile;
use League\Csv\Reader;
use Mtc\Shop\Brand;
use Mtc\Shop\Category;

/**
 * Seo csv class
 *
 * @author Lukas Giegerich | mtc.
 * @version 2014-01-20
 */
class Csv
{
    /**
     * echoes a csv of the given type (paths, brands or cats) to use for dl
     *
     * @param string $type
     */
    public static function csvPrint($type)
    {
        if ($type === 'paths') {
            $headers = (new Page())->getFillable();
            $query = Page::exportQuery();
        } elseif ($type === 'brands') {
            $headers = [
                'id',
                'name',
                'seo_title',
                'seo_description',
            ];
            $query = Brand::query()
                ->select($headers)
                ->orderBy('id');
        } elseif ($type === 'categories') {
            $headers = [
                'id',
                'name',
                'seo_title',
                'seo_description',
            ];
            $query = Category::query()
                ->select($headers)
                ->orderBy('id');
        } elseif ($type === 'headings') {
            $headers = (new Heading())->getFillable();
            $query = Heading::exportQuery();
        } elseif ($type === 'redirects') {
            $headers = (new Redirect())->getFillable();
            $query = Redirect::exportQuery();
        }

        if (empty($query)) {
            return null;
        }

        $data = $query->get();
        $data->prepend($headers);
        return self::outPutCsv($data->toArray());
    }

    /**
     * The joy of fputcsv() returning a string
     *
     * @param array $array
     * @return string
     */
    private static function outPutCsv($array)
    {
        ob_start();
        $fp = fopen('php://output', 'w');
        foreach ($array as $entry) {
            fputcsv($fp, $entry);
        }
        fclose($fp);
        return ob_get_clean();
    }

    /**
     * @param Request $request
     * @param $type
     * @throws \Exception
     */
    public static function importCsv(Request $request, $type)
    {
        /** @var UploadedFile $data */
        $uploaded_file = $request->file('csv');

        $csv = Reader::createFromPath($uploaded_file->getRealPath(), 'r');
        $csv->setHeaderOffset(0);

        $header = $csv->getHeader();
        $records = $csv->getRecords($header);

        switch ($type) {
            case 'paths':
                self::importPages($records);
                break;
            case 'brands':
                self::importBrands($records);
                break;
            case 'categories':
                self::importCategories($records);
                break;
            case 'headings':
                self::importHeadings($records);
                break;
            case 'redirects':
                self::importRedirects($records);
                break;
            default:
                throw new \Exception('Unknown import type');
                break;
        }
    }

    /**
     * Update Page data with import data
     *
     * @param $data
     */
    public static function importPages($data)
    {
        foreach ($data as $entry) {
            Page::query()
                ->updateOrCreate([
                    'path' => $entry['path']
                ], [
                    'path' => $entry['path'],
                    'title' => $entry['title'],
                    'description' => $entry['description'],
                ]);
        }
    }

    /**
     * Update Brand data with import data
     * @param $data
     */
    public static function importBrands($data)
    {
        foreach ($data as $entry) {
            Brand::query()
                ->where('id', $entry['id'])
                ->update([
                    'seo_title' => $entry['seo_title'],
                    'seo_description' => $entry['seo_description'],
                ]);
        }
    }

    /**
     * Update category info with import data
     *
     * @param $data
     */
    public static function importCategories($data)
    {
        foreach ($data as $entry) {
            Category::query()
                ->where('id', $entry['id'])
                ->update([
                    'seo_title' => $entry['seo_title'],
                    'seo_description' => $entry['seo_description'],
                ]);
        }
    }

    /**
     * Update DB with the redirect data
     *
     * @param $data
     */
    public static function importRedirects($data)
    {
        foreach ($data as $entry) {
            Redirect::query()
                ->updateOrCreate([
                    'old_url' => $entry['old_url'],
                    'new_url' => $entry['new_url'],
                ], [
                    'old_url' => $entry['old_url'],
                    'new_url' => $entry['new_url'],
                    'code' => $entry['code'],
                ]);
        }
    }

    /**
     * Update DB with the redirect data
     *
     * @param $data
     */
    public static function importHeadings($data)
    {
        foreach ($data as $entry) {
            Heading::query()
                ->updateOrCreate([
                    'path' => $entry['path'],
                ], [
                    'path' => $entry['path'],
                    'text' => $entry['text'],
                ]);
        }
    }
}
