<?php

namespace Mtc\Coupons\Charts;

use Carbon\Carbon;
use ConsoleTVs\Charts\Classes\Chartjs\Chart;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\DB;
use Illuminate\Http\Request;

class Coupon extends Chart
{
    /**
     * @var \Mtc\Coupons\Coupon
     */
    protected $coupon;

    /**
     * @var Request
     */
    protected $request;

    /**
     * @var Carbon
     */
    protected $start_date;

    /**
     * @var Carbon
     */
    protected $end_date;

    /**
     * Initializes the chart.
     *
     * @return void
     */
    public function __construct(\Mtc\Coupons\Coupon $coupon, Request $request)
    {
        parent::__construct();
        $this->coupon = $coupon;
        $this->request = $request;

        $this->processRequest();

        $this->loadOptions();
        $this->loadLabels();
        $this->loadOrdersData();
    }

    /**
     * Processt request
     */
    protected function processRequest()
    {
        $this->start_date = $this->request->has('chart_start_date') ? Carbon::createFromFormat('d/m/Y', $this->request->input('chart_start_date'))->startOfDay() : Carbon::now()->startOfDay()->subMonth();
        $this->end_date = $this->request->has('chart_end_date') ? Carbon::createFromFormat('d/m/Y', $this->request->input('chart_end_date')) : Carbon::now();
    }

    /**
     * Load chart options
     */
    protected function loadOptions()
    {
        $this->options([
            'scales' => [
                'yAxes' => [
                    [
                        'ticks' => [
                            'stepSize' => config('coupons.chart.step_size', 1)
                        ],
                    ],
                ],
            ],
        ]);
    }

    /**
     * Load labels for chart
     */
    protected function loadLabels()
    {
        $labels = [];
        $start_date = Carbon::parse($this->start_date);

        if ($this->request->input('chart_group_by') === 1) {
            $start_date = $start_date->startOfMonth();
        } elseif ($this->request->input('chart_group_by') === 2) {
            $start_date = $start_date->startOfYear();
        }

        while ($start_date <= $this->end_date) {
            $labels[] = $this->getLabelName($start_date);
            $start_date = $this->updateStartDate($start_date);
        }
        $this->labels($labels);
    }

    /**
     * @param Carbon $date
     * @return int|string
     */
    protected function getLabelName(Carbon $date)
    {
        if ($this->request->input('chart_group_by') === '1') {
            //return $date->format('F');
            return $date->month;
        }

        if ($this->request->input('chart_group_by') === '2') {
            //return $date->format('F');
            return $date->year;
        }

        return $date->toDateString();
    }

    /**
     * @param Carbon $date
     * @return Carbon
     */
    protected function updateStartDate(Carbon $date) : Carbon
    {
        if ($this->request->input('chart_group_by') === '1') {
            return $date->addMonth();
        }
        if ($this->request->input('chart_group_by') === '2') {
            return $date->addYear();
        }

        return $date->addDay();
    }

    /**
     * Load orders data
     */
    protected function loadOrdersData()
    {
        $discounts_orders_usage = App::make(config('orders.order_class'))
            ->where('paid_at', '>=', $this->start_date)
            ->where('paid_at', '<=', $this->end_date)
            ->whereHas('discounts', function($query) {
                $query->where('discount_type', 'coupon')
                    ->where('discount_id', $this->coupon->id);
            })
            ->select(['*', DB::raw($this->getDataSelect())])
            ->get()
            ->groupBy('date')
            ->map(function ($orders) {
                $orders_cost_total = $orders->sum(function($order) {
                    return $order->cost_total->original();
                });
                return [
                    'orders_count' => $orders->count(),
                    'orders_cost_total' => round($orders_cost_total, 2)
                ];
            });

        $data_orders_count = [];
        $data_orders_cost_total = [];
        foreach ($this->labels as $label) {
            $data_orders_count[] = $discounts_orders_usage[$label]['orders_count'] ?? 0;
            $data_orders_cost_total[] = $discounts_orders_usage[$label]['orders_cost_total'] ?? 0;
        }

        $this->dataset('Orders', 'line', $data_orders_count);
        $this->dataset('Orders cost total', 'line', $data_orders_cost_total)
            ->options([
                'backgroundColor' => '#ccccff',
                ]);
    }

    /**
     * @return string
     */
    protected function getDataSelect() : string
    {
        if ($this->request->input('chart_group_by') === '1') {
            return 'MONTH(paid_at) as date';
        }

        if ($this->request->input('chart_group_by') === '2') {
            return 'YEAR(paid_at) as date';
        }

        return 'DATE(paid_at) as date';
    }
}
