<?php

namespace Mtc\Coupons\Modifiers;

use Mtc\Basket\Contracts\BasketRepositoryInterface;
use Mtc\ShippingManager\Contracts\ShippingModifier;
use Mtc\ShippingManager\TableRate;

/**
 * Class FreeWorldwideDelivery
 *
 * Modifier that applies free world-wide delivery when relevant coupon is added to basket
 */
class FreeWorldwideDelivery implements ShippingModifier
{
    /**
     * Retrieve the name
     *
     * @return string
     */
    public function getName()
    {
        return 'Free Worldwide Delivery Coupon';
    }

    /**
     * Check whether the modifier is applicable to the rate
     *
     * @return bool
     */
    public function isApplicable(BasketRepositoryInterface $basket_repository, TableRate $rate)
    {
        $modifier_name = array_search(self::class, config('coupons.shipping_modifiers'));

        // Not applied to this rate
        if (!in_array($modifier_name, $rate->modifiers)) {
            return false;
        }

        // No Coupon - is not relevant to basket
        if (!$basket_repository->hasDiscount(0, 'coupon')) {
            return false;
        }

        $basket = $basket_repository->getModel();
        $is_applicable = collect($basket->discounts->where('discount_type', 'coupon'))
            ->filter(function ($basket_discount) use ($modifier_name) {
                return $basket_discount->discount->shipping_modifier === $modifier_name;
            })
            ->isNotEmpty();

        // Coupon does not have this modifier
        return $is_applicable;
    }

    /**
     * Define whether this modifier MUST be matched for this rate to work
     *
     * @return bool
     */
    public function isMandatoryOnRate(): bool
    {
        return false;
    }

    /**
     * Define whether this modifier matching will discard other rates
     *
     * @return bool
     */
    public function willDiscardOtherRates(): bool
    {
        return true;
    }

    /**
     * Define whether this modifier matching will ignore the discard of other methods
     *
     * @return bool
     */
    public function isGloballyAvailable(): bool
    {
        return true;
    }
}
