<?php

namespace Mtc\Coupons\Providers;

use Illuminate\Database\Eloquent\Relations\Relation;
use Illuminate\Support\Facades\App;
use Illuminate\Support\ServiceProvider;
use Mtc\Coupons\Console\Commands\InstallCoupons;
use Mtc\Coupons\Contracts\CouponRepositoryInterface;
use Mtc\ShippingManager\Facades\ShippingModifiers;

/**
 * Class CouponServiceProvider
 *
 * @package Mtc\Coupons\Providers
 */
class CouponServiceProvider extends ServiceProvider
{
    /**
     * Bootstrap the application services.
     */
    public function boot()
    {
        //Check if package enabled
        if (config('coupons.enabled') !== true) {
            return;
        }

        Relation::morphMap([
            'coupon' => config('coupons.coupon_class')
        ]);

        // Load translations
        $this->loadTranslationsFrom(dirname(__DIR__, 2) . '/resources/lang', 'coupons');

        if ($this->app->runningInConsole()) {
            $this->loadMigrationsFrom(dirname(__DIR__, 2) . '/database/migrations');
            $this->bootResourcePublish();

        } else {
            app('twig_loader')->addPath(dirname(__DIR__, 2) . '/resources/templates');
        }

        if (class_exists(ShippingModifiers::class)) {
            collect(config('coupons.shipping_modifiers'))
                ->each(function ($class, $name) {
                    ShippingModifiers::register($name, $this->app->make($class));
                });
        }
    }

    /**
     * Register the application services.
     */
    public function register()
    {
        // Automatically apply the package configuration
        $this->mergeConfigFrom(dirname(__DIR__, 2) . '/config/coupons.php', 'coupons');

        //Check if package enabled
        if (config('coupons.enabled', false) !== true) {
            return;
        }

        collect(config('coupons.service_providers', []))
            ->each(function ($service_provider) {
                $this->app->register($service_provider);
            });

        $this->registerBindings();
    }

    /**
     * Register bindings used by the package
     */
    protected function registerBindings()
    {
        // Load a coupon repository
        $this->app->singleton(CouponRepositoryInterface::class, function ($app) {
            return $this->app->make(config('coupons.repository_class'));
        });

        // Register the validator class to use with the facade
        $this->app->singleton('coupon', function () {
            return $this->app->make(config('coupons.coupon_class'));
        });

        // Register the restriction class to use with the facade
        $this->app->singleton('coupon-restriction', function () {
            return $this->app->make(config('coupons.restriction_class'));
        });

        // Register the validator class to use with the facade
        $this->app->singleton('coupon-validator', function () {
            $class = config('coupons.validator_class');
            return new $class(App::make('basket'));
        });
    }

    /**
     * Register resources that get published
     */
    protected function bootResourcePublish()
    {
        $this->publishes([
            dirname(__DIR__, 2) . '/config/' => config_path()
        ], 'config');

        $this->publishes([
            dirname(__DIR__, 2) . '/database/seeds/' => database_path('seeds')
        ], 'seeds');

        $this->publishes([
            dirname(__DIR__, 2) . '/resources/js/' => resource_path('js/vendor')
        ], 'assets');

        $this->publishes([
            dirname(__DIR__, 2) . '/resources/less/' => resource_path('less/vendor')
        ], 'assets');
    }
}
