<?php

namespace Mtc\Crm;

use Illuminate\Http\Request;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Mtc\Crm\Contracts\FormActionModel;
use Mtc\Crm\Contracts\FormModel;
use Mtc\Crm\Contracts\FormQuestionModel;
use Mtc\Crm\Contracts\FormSectionModel;

class FormRepository
{
    /**
     * Update form from request
     *
     * @param Request $request
     * @param FormModel $form
     * @return void
     */
    public function update(Request $request, FormModel $form)
    {
        DB::beginTransaction();
        $form->update([
            'type_id' => $request->input('type_id'),
            'initial_status_id' => $request->input('initial_status_id'),
            'initial_assignee' => $request->input('initial_assignee'),
            'success_message' => $request->input('success_message'),
            'send_customer_confirmation_email' => $request->input('send_customer_confirmation_email') ?? true,
        ]);

        $this->syncActions(collect($request->input('actions', [])), $form);
        DB::commit();
    }

    /**
     * Sync actions of the form
     *
     * @param Collection $actions
     * @param FormModel $form
     * @return void
     */
    protected function syncActions(Collection $actions, FormModel $form)
    {
        $enabled_forms = $actions
            ->filter(fn($action) => $action['enabled'] ?? false);

        $form->actions()
            ->whereNotIn('action_name', $enabled_forms->keys())
            ->delete();

        $enabled_forms->each(fn ($fields, $name) => $form->actions()
            ->updateOrCreate([
                'action_name' => $name,
            ], [
                'fields' => $fields,
            ]));
    }

    public function delete(FormModel $form): void
    {
        $form->sections()->delete();
        $form->actions()->delete();
        $form->questions()->delete();
        $form->delete();
    }

    public function store(Request $request): FormModel
    {
        return App::make(FormModel::class)
            ->newQuery()
            ->create([
                'name' => $request->input('name'),
                'type_id' => config('crm.default_type_id'),
                'send_customer_confirmation_email' => true,
                'initial_status_id' => config('crm.default_status_id'),
            ]);
    }

    public function copy(FormModel $form, string $title, bool $withContent = false): FormModel
    {
        $form->load([
            'actions',
            'sections.questions',
        ]);
        $data = $withContent
            ? $form->toArray()
            : $form->only([
                'type_id',
                'is_active',
                'active_from',
                'active_until',
                'actions',
                'initial_status_id',
                'initial_assignee',
            ]);

        $data['name'] = $title;

        /** @var FormModel $copy */
        $copy = $form->newQuery()->create($data);

        $form->actions->each(fn(FormActionModel $action) => $copy->actions()->create($action->toArray()));
        $form->sections->each(fn(FormSectionModel $section) => $this->copySection($copy, $section));

        return $copy;
    }

    private function copySection(FormModel $copy, FormSectionModel $section): void
    {
        /** @var FormSectionModel $sectionCopy */
        $sectionCopy = $copy->sections()->create($section->toArray());
        $section->questions
            ->each(function (FormQuestionModel $question) use ($sectionCopy) {
                $data = $question->toArray();
                $data['form_id'] = $sectionCopy->form_id;
                $data['section_id'] = $sectionCopy->section_id;
                $sectionCopy->questions()->create($data);
            });
    }
}
