<?php

namespace Mtc\Crm\Jobs;

use Carbon\Carbon;
use Exception;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Mail;
use Mtc\Crm\Contracts\EnquiryAction;
use Mtc\Crm\Contracts\EnquiryActionModel;
use Mtc\Crm\Facades\Enquiries;
use Mtc\Crm\Contracts\EnquiryModel;

class ProcessIncomingEnquiryJob implements ShouldQueue
{
    use Dispatchable;
    use InteractsWithQueue;
    use Queueable;
    use SerializesModels;

    /**
     * Create a new job instance.
     *
     * @return void
     */
    public function __construct(protected EnquiryModel $enquiry)
    {
        $this->onQueue('emails');
    }

    /**
     * Execute the job.
     *
     * @return void
     */
    public function handle()
    {
        if ($this->process()) {
            $this->enquiry->refresh()->update([
                'ingested_at' => Carbon::now()
            ]);
        }
    }

    private function process(): bool
    {
        $this->sendConfirmation();

        $actions = $this->enquiry
            ->actions()
            ->whereNull('processed_at')
            ->where('attempts', '<', config('crm.enquiry_action_max_attempts'))
            ->get();

        if (!$actions->count()) {
            return true;
        }

        // reject all successes, return success if no failures in list
        return collect($actions)
            ->reject(fn(EnquiryActionModel $action) => $this->handleEnquiryAction($action))
            ->isEmpty();
    }

    private function handleEnquiryAction(EnquiryActionModel $action): bool
    {
        try {
            $handler = Enquiries::makeAction($action->action_name);
            $result = $handler->handle($action, $this->enquiry);
            match ($result) {
                false => $this->markFailed($action, $handler),
                true => $this->markSuccessful($action, $handler),
                default => null,
            };
            return $result ?? true;
        } catch (Exception $exception) {
            Log::error('Enquiry action failed', [
                'enquiry' => $this->enquiry,
                'action' => $action,
                'error' => $exception,
            ]);
            $action->update([
                'failed_at' => Carbon::now(),
                'context' => 'Failed due to a system error'
            ]);
            $action->increment('attempts');
            return false;
        }
    }

    private function sendConfirmation(): void
    {
        if ($this->enquiry->confirmation_sent) {
            return;
        }
        if (empty($this->enquiry->form->send_customer_confirmation_email)) {
            $this->enquiry->refresh()->update([
                'confirmation_sent' => true,
            ]);
            return;
        }

        try {
            $recipient = $this->enquiry->email;
            if (empty($recipient) || !filter_var($recipient, FILTER_VALIDATE_EMAIL)) {
                return;
            }
            $class = config('crm.enquiry_confirmation_mail');
            Mail::to($this->enquiry->email)->send(new $class($this->enquiry));
            $this->enquiry->refresh()->update([
                'confirmation_sent' => true,
            ]);
        } catch (\Exception $exception) {
            Log::warning($exception->getMessage());
        }
    }

    private function markFailed(EnquiryActionModel $action, EnquiryAction $handler): void
    {
        $action->update([
            'failed_at' => Carbon::now(),
            'context' => 'Failed with reason: ' . $handler->failureReason()
        ]);
        $action->increment('attempts');
    }

    private function markSuccessful(EnquiryActionModel $action, EnquiryAction $handler): void
    {
        $action->update([
            'processed_at' => Carbon::now(),
            'context' => $handler->successDetails(),
        ]);
    }
}
