<?php

namespace Mtc\Crm\Tests\Feature;

use Carbon\Carbon;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Notification;
use Mtc\Crm\Mail\NewEnquiryMail;
use Mtc\Crm\Models\Enquiry;
use Mtc\Crm\Models\EnquiryAction;
use Mtc\Crm\Models\EnquiryStatus;
use Mtc\Crm\Models\EnquiryTag;
use Mtc\Crm\Models\EnquiryType;
use Mtc\Crm\Tests\AsUser;
use Mtc\Crm\Tests\TestCase;
use Mtc\Crm\Tests\User;

class EnquiryControllerTest extends TestCase
{
    use RefreshDatabase;
    use AsUser;

    public function testIndex()
    {
        $this->asUser();
        $response = $this->get(route('enquiries.index'));

        $this->assertEquals(200, $response->status());
        $body = $response->getData(true);
        $this->assertArrayHasKey('current_page', $body);
        $this->assertArrayHasKey('data', $body);
        $this->assertEquals(0, count($body['data']));

        Enquiry::factory()->create(['title' => 'foo']);
        Enquiry::factory()->create(['title' => 'baz']);
        Enquiry::factory()->create(['title' => 'bar']);

        $response = $this->get(route('enquiries.index'));

        $body = $response->getData(true);
        $this->assertEquals(3, count($body['data']));
    }

    public function testArchiveIndex()
    {
        $this->asUser();
        Enquiry::factory()->create(['title' => 'foo']);
        Enquiry::factory()->create(['title' => 'baz']);
        Enquiry::factory()->create(['title' => 'bar', 'deleted_at' => Carbon::now()]);

        $response = $this->get(route('enquiries.index', ['archive' => 1]));

        $body = $response->getData(true);
        $this->assertEquals(1, count($body['data']));
    }

    public function testFilters()
    {
        $this->asUser();
        EnquiryStatus::factory()->create();
        EnquiryTag::factory()->create();
        EnquiryTag::factory()->create();
        EnquiryType::factory()->create();
        EnquiryType::factory()->create();
        EnquiryType::factory()->create();
        $response = $this->get(route('enquiries.filters'));
        $this->assertEquals(200, $response->status());

        $body = $response->getData(true);
        $this->assertIsArray($body);
        $this->assertArrayHasKey('types', $body);
        $this->assertArrayHasKey('tags', $body);
        $this->assertArrayHasKey('statuses', $body);
        $this->assertEquals(3, count($body['types']['data']));
        $this->assertEquals(2, count($body['tags']));
        $this->assertEquals(1, count($body['statuses']['data']));
    }

    public function testDetails()
    {
        $this->asUser();
        $enquiry = Enquiry::factory()->create(['title' => 'foo']);
        $response = $this->get(route('enquiries.details', $enquiry->id));
        $this->assertEquals(200, $response->status());

        $data = $response->getData(true);
        $this->assertArrayHasKey('subscribers', $data);
        $this->assertArrayHasKey('last_interaction_diff', $data);
        $this->assertArrayHasKey('id', $data);
        $this->assertArrayHasKey('title', $data);
        $this->assertArrayHasKey('assignee', $data);
        $this->assertArrayHasKey('messages', $data);
    }

    public function testShow()
    {
        $this->asUser();
        $enquiry = Enquiry::factory()->create(['title' => 'foo']);
        $response = $this->get(route('enquiries.show', $enquiry->id));
        $this->assertEquals(200, $response->status());

        $data = $response->getData(true);
        $this->assertArrayHasKey('enquiry', $data);
        $this->assertArrayHasKey('id', $data['enquiry']);
        $this->assertArrayHasKey('title', $data['enquiry']);
        $this->assertArrayHasKey('statuses', $data);
        $this->assertArrayHasKey('managers', $data);
    }

    public function testAddMessage()
    {
        Notification::fake();
        Mail::fake();
        $this->asUser();
        $enquiry = Enquiry::factory()->create(['title' => 'foo']);
        $response = $this->postJson(route('enquiries.addMessage', $enquiry->id), [
            'message' => 'lorem ipsum'
        ]);
        $this->assertEquals(200, $response->status());

        $this->assertTrue($enquiry->messages()->where('message', 'lorem ipsum')->exists());
    }

    public function testAddMessageWithStatusChagne()
    {
        Notification::fake();
        Mail::fake();
        $this->asUser();
        $enquiry = Enquiry::factory()->create(['title' => 'foo']);
        $response = $this->postJson(route('enquiries.addMessage', $enquiry->id), [
            'message' => 'lorem ipsum',
            'new_status' => 10,
        ]);
        $this->assertEquals(200, $response->status());

        $enquiry->refresh();
        $this->assertTrue($enquiry->messages()->where('message', 'lorem ipsum')->exists());
        $this->assertEquals(10, $enquiry->status_id);
    }

    public function testAddTag()
    {
        $this->asUser();
        $enquiry = Enquiry::factory()->create(['title' => 'foo']);
        $tag = EnquiryTag::factory()->create();

        $this->postJson(route('enquiries.add-tag', [$enquiry->id, $tag->id]));
        $this->assertTrue($enquiry->tags()->where('enquiry_tag_id', $tag->id)->exists());
    }

    public function testRemoveTag()
    {
        $this->asUser();
        $enquiry = Enquiry::factory()->create(['title' => 'foo']);
        $tag = EnquiryTag::factory()->create();

        $enquiry->tags()->attach($tag->id);

        $this->postJson(route('enquiries.remove-tag', [$enquiry->id, $tag->id]));
        $this->assertFalse($enquiry->tags()->where('enquiry_tag_id', $tag->id)->exists());
    }

    public function testSetStatus()
    {
        $this->asUser();
        $enquiry = Enquiry::factory()->create(['title' => 'foo', 'status_id' => 99]);
        $status = EnquiryStatus::factory()->create();

        $this->assertNotEquals($status->id, $enquiry->status_id);
        $response = $this->postJson(route('enquiries.set-status', $enquiry->id), [
            'status' => $status->id
        ]);

        $this->assertEquals(200, $response->getStatusCode());
        $enquiry->refresh();
        $this->assertEquals($status->id, $enquiry->status_id);
    }


    public function testSetNonExistingStatus()
    {
        $this->asUser();
        $enquiry = Enquiry::factory()->create(['title' => 'foo', 'status_id' => 99]);

        $response = $this->postJson(route('enquiries.set-status', $enquiry->id), [
            'status' => 999
        ]);

        $this->assertEquals(422, $response->getStatusCode());
    }

    public function testAssign()
    {
        Notification::fake();
        $this->asUser();
        $enquiry = Enquiry::factory()->create(['title' => 'foo', 'status_id' => 99]);
        $user = User::query()->create(['name' => 'peter', 'email' => 'peter@example.com', 'password' => '']);

        $response = $this->postJson(route('enquiries.assign', $enquiry->id), [
            'user' => $user->id
        ]);

        $this->assertEquals(200, $response->getStatusCode());
        $enquiry->refresh();
        $this->assertEquals($user->id, $enquiry->assigned_user_id);
    }

    public function testSubscribe()
    {
        Notification::fake();
        $this->asUser();

        $enquiry = Enquiry::factory()->create(['title' => 'foo', 'status_id' => 99]);
        $user = User::query()->create(['name' => 'peter', 'email' => 'peter@example.com', 'password' => '']);

        $response = $this->postJson(route('enquiries.subscribers.store', $enquiry->id), [
            'subscriber' => $user->id
        ]);

        $this->assertEquals(200, $response->getStatusCode());
        $this->assertTrue($enquiry->subscribers()->where('user_id', $user->id)->exists());
    }

    public function testUnsubscribe()
    {
        $this->asUser();

        $enquiry = Enquiry::factory()->create(['title' => 'foo', 'status_id' => 99]);
        $user = User::query()->create(['name' => 'peter', 'email' => 'peter@example.com', 'password' => '']);

        $enquiry->subscribers()->attach($user->id);
        $this->assertTrue($enquiry->subscribers()->where('user_id', $user->id)->exists());
        $response = $this->deleteJson(route('enquiries.subscribers.destroy', [$enquiry->id, $user->id]));

        $this->assertEquals(200, $response->getStatusCode());
        $this->assertFalse($enquiry->subscribers()->where('user_id', $user->id)->exists());
    }

    public function testUsersToSubscribe()
    {
        $this->asUser();
        $response = $this->get(route('enquiries.users-to-subscribe'));
        $this->assertEquals(200, $response->status());
        $this->assertEquals(1, count($response->getData(true)));
    }

    public function testResubmitEnquiryAction()
    {
        $this->asUser();
        $enquiry = Enquiry::factory()->create();
        $action = EnquiryAction::factory()->create([
            'enquiry_id' => $enquiry->id,
            'action_name' => 'send-copy-via-mail',
            'data' => [
                'recipient' => 'john@example.com',
                'enquiry_subject' => 'foo',
                'enquiry_content' => 'lorem ipsum'
            ],
            'processed_at' => null,
            'failed_at' => Carbon::now(),
        ]);

        Mail::fake();
        $response = $this->postJson(route('enquiries.resubmit-action', [$enquiry, $action]));

        $this->assertEquals(200, $response->getStatusCode());
        $this->assertArrayHasKey('processed_at', $response->json());
        Mail::assertSent(NewEnquiryMail::class, function (NewEnquiryMail $mail) {
            return $mail->hasTo('john@example.com');
        });
    }

    public function testResubmitEnquiryActionAleadyDone()
    {
        $this->asUser();
        $enquiry = Enquiry::factory()->create();
        $action = EnquiryAction::factory()->create([
            'enquiry_id' => $enquiry->id,
            'action_name' => 'send-copy-via-mail',
            'data' => [
                'recipient' => 'john@example.com',
                'enquiry_subject' => 'foo',
                'enquiry_content' => 'lorem ipsum'
            ],
            'processed_at' => Carbon::now(),
            'failed_at' => null,
        ]);

        $response = $this->postJson(route('enquiries.resubmit-action', [$enquiry, $action]));

        $this->assertEquals(200, $response->getStatusCode());
        $this->assertArrayHasKey('processed_at', $response->json());
        $this->assertEquals('Already Processed', $response->json('context'));
    }

    public function testResubmitEnquiryActionFailingScenario()
    {
        $this->asUser();
        $enquiry = Enquiry::factory()->create();
        $action = EnquiryAction::factory()->create([
            'enquiry_id' => $enquiry->id,
            'action_name' => 'send-copy-via-mail',
            'data' => [
                'recipient' => '',
                'enquiry_subject' => 'foo',
                'enquiry_content' => 'lorem ipsum'
            ],
            'processed_at' => null,
            'failed_at' => Carbon::now(),
        ]);

        $response = $this->postJson(route('enquiries.resubmit-action', [$enquiry, $action]));

        $this->assertEquals(200, $response->getStatusCode());
        $this->assertArrayHasKey('failed_at', $response->json());
    }
}
