<?php

namespace Mtc\Crm;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Collection;
use Mtc\Crm\Contracts\EnquiryStatusModel;
use Mtc\Crm\Contracts\EnquiryTagModel;
use Mtc\Crm\Contracts\EnquiryTypeModel;

class EnquiryDataRepository
{
    public function __construct(
        protected EnquiryTypeModel $type_model,
        protected EnquiryStatusModel $status_model,
        protected EnquiryTagModel $tag_model,
    ) {
        //
    }

    /**
     * Get available tags
     *
     * @return Collection
     */
    public function getTags(): Collection
    {
        return $this->tag_model->newQuery()
            ->latest()
            ->get();
    }

    /**
     * Find tag instance
     *
     * @param int $id
     * @return EnquiryTagModel|Model
     */
    public function findTag(int $id): EnquiryTagModel
    {
        return $this->tag_model->newQuery()
            ->findOrFail($id);
    }

    /**
     * Add tag record
     *
     * @param string $name
     * @return Model
     */
    public function addTag(string $name): Model
    {
        return $this->tag_model->newQuery()
            ->create([
                'name' => $name
            ]);
    }

    /**
     * Update tag record
     *
     * @param int $id
     * @param string $name
     * @return int
     */
    public function updateTag(int $id, string $name)
    {
        return $this->tag_model->newQuery()
            ->where('id', $id)
            ->update([
                'name' => $name,
            ]);
    }

    /**
     * Remove tag record
     *
     * @param int $id
     * @return bool|mixed|null
     */
    public function removeTag(int $id)
    {
        return $this->tag_model->newQuery()
            ->find($id)
            ?->delete();
    }

    /**
     * Get model object for status
     *
     * @return Model
     */
    public function getStatusModel(): Model
    {
        return $this->status_model;
    }

    /**
     * Get list of available statuses
     *
     * @return Collection
     */
    public function getStatuses(): Collection
    {
        return $this->status_model->newQuery()
            ->latest()
            ->get();
    }

    /**
     * Add new status record
     *
     * @param string $name
     * @param bool $is_final
     * @return Model
     */
    public function addStatus(string $name, bool $is_final): Model
    {
        return $this->status_model->newQuery()
            ->create([
                'name' => $name,
                'is_final' => $is_final,
            ]);
    }

    /**
     * Update status record
     *
     * @param int $id
     * @param string $name
     * @param bool $is_final
     * @return int
     */
    public function updateStatus(int $id, string $name, bool $is_final)
    {
        return $this->status_model->newQuery()
            ->where('id', $id)
            ->update([
                'name' => $name,
                'is_final' => $is_final,
            ]);
    }

    /**
     * Find status instance
     *
     * @param int $id
     * @return EnquiryStatusModel|Model
     */
    public function findStatus(int $id): EnquiryStatusModel
    {
        return $this->status_model->newQuery()
            ->findOrFail($id);
    }

    /**
     * Remove status
     *
     * @param int $id
     * @return bool|mixed|null
     */
    public function removeStatus(int $id)
    {
        return $this->status_model->newQuery()
            ->findOrFail($id)
            ?->delete();
    }

    /**
     * Get model object for type
     *
     * @return Model
     */
    public function getTypeModel(): Model
    {
        return $this->type_model;
    }

    /**
     * Get list of available types
     *
     * @return Collection
     */
    public function getTypes(): Collection
    {
        return $this->type_model->newQuery()
            ->latest()
            ->get();
    }

    /**
     * Find type instance
     *
     * @param int $id
     * @return EnquiryTypeModel|Model
     */
    public function findType(int $id): EnquiryTypeModel
    {
        return $this->type_model->newQuery()
            ->findOrFail($id);
    }

    /**
     * Add new type
     *
     * @param string $name
     * @return Model
     */
    public function addType(string $name): Model
    {
        return $this->type_model->newQuery()
            ->create([
                'name' => $name
            ]);
    }

    /**
     * Update type
     *
     * @param int $id
     * @param string $name
     * @return int
     */
    public function updateType(int $id, string $name)
    {
        return $this->type_model->newQuery()
            ->where('id', $id)
            ->update([
                'name' => $name,
            ]);
    }

    /**
     * Remove a type instance
     *
     * @param int $id
     * @return bool|mixed|null
     */
    public function removeType(int $id)
    {
        return $this->type_model->newQuery()
            ->findOrFail($id)
            ?->delete();
    }
}
