<?php

namespace Mtc\Crm\Contracts;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\MorphTo;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Support\Facades\Config;
use Mtc\Crm\Factories\EnquiryFactory;
use Mtc\Crm\Scopes\WithStatusScope;
use Mtc\Crm\Traits\ModelSortAndFilter;

/**
 * @param int $reason_id
 * @param string $reason_type
 * @param int $form_id
 * @param string $email
 * @param int $submitter_id
 * @param string $submitter_type
 * @param int $type_id
 * @param int $status_id
 * @param string $title
 * @param string $text
 * @param array $details
 * @param array $data
 * @param int $assigned_user_id
 * @param int $customer_id
 * @param string $status_text
 * @param bool $is_closed
 */
abstract class EnquiryModel extends Model
{
    use SoftDeletes;
    use HasFactory;
    use ModelSortAndFilter;

    /**
     * @var string
     */
    protected $table = 'enquiries';

    /**
     * @var string[]
     */
    protected $guarded = [
        'id'
    ];

    protected $casts = [
        'details' => 'array',
        'data' => 'array'
    ];

    protected $appends = [
        'status_text',
        'is_closed',
    ];

    /**
     * Boot model
     */
    protected static function boot()
    {
        parent::boot();

        self::addGlobalScope(new WithStatusScope());
    }

    /**
     * Model Factory
     *
     * @return EnquiryFactory
     */
    protected static function newFactory()
    {
        return EnquiryFactory::new();
    }

    /**
     * Relationship with reason of the enquiry (attached data model like vehicle, offer, page, product)
     *
     * @return MorphTo
     */
    public function reason(): MorphTo
    {
        return $this->morphTo('reason');
    }

    /**
     * @return BelongsTo
     */
    public function form(): BelongsTo
    {
        return $this->belongsTo(Config::get('crm.form_model'), 'form_id');
    }

    /**
     * @return BelongsTo
     */
    public function status(): BelongsTo
    {
        return $this->belongsTo(Config::get('crm.status_model'), 'status_id');
    }

    /**
     * @return HasMany
     */
    public function messages(): HasMany
    {
        return $this->hasMany(Config::get('crm.message_model'));
    }


    /**
     * @return HasMany
     */
    public function actions(): HasMany
    {
        return $this->hasMany(Config::get('crm.enquiry_action_model'));
    }

    /**
     * @return HasMany
     */
    public function statusHistory(): HasMany
    {
        return $this->hasMany(Config::get('crm.status_history_model'));
    }

    /**
     * @return BelongsToMany
     */
    public function subscribers(): BelongsToMany
    {
        return $this->belongsToMany(Config::get('auth.providers.users.model'), 'enquiry_subscribers');
    }

    /**
     * @return BelongsTo
     */
    public function assignee(): BelongsTo
    {
        return $this->belongsTo(Config::get('auth.providers.users.model'), 'assigned_user_id');
    }

    /**
     * @return MorphTo
     */
    public function submitter(): MorphTo
    {
        return $this->morphTo('submitter');
    }

    /**
     * @return BelongsTo
     */
    public function type(): BelongsTo
    {
        return $this->belongsTo(Config::get('crm.type_model'));
    }

    /**
     * @return BelongsToMany
     */
    public function tags(): BelongsToMany
    {
        return $this->belongsToMany(Config::get('crm.tag_model'), 'enquiry_assigned_tags');
    }

    /**
     * Scope for getting current users enquiries
     *
     * @param Builder $query
     * @return void
     */
    public function scopeMine(Builder $query)
    {
        $query->where('assigned_user_id', auth()->id());
    }

    /**
     * status_text
     *
     * @return string
     */
    public function getStatusTextAttribute(): string
    {
        return $this->status->name ?? '';
    }

    /**
     * is_closed - whether status can be changed
     *
     * @return string
     */
    public function getIsClosedAttribute(): string
    {
        return $this->status->is_final ?? 0;
    }

    /**
     * Only trashed / not trashed scope
     *
     * @param Builder $builder
     * @param bool $trashed
     * @return Builder
     */
    public function scopeArchived(Builder $builder, bool $trashed = true): Builder
    {
        return $trashed
            ? $builder->onlyTrashed()
            : $builder;
    }
}
