<?php

namespace Mtc\Crm\Tests\Feature;

use Illuminate\Foundation\Testing\RefreshDatabase;
use Mtc\Crm\Models\EnquiryStatus;
use Mtc\Crm\Seeds\EnquiryStatusSeeder;
use Mtc\Crm\Tests\AsUser;
use Mtc\Crm\Tests\TestCase;

class EnquiryStatusControllerTest extends TestCase
{
    use RefreshDatabase;
    use AsUser;

    public function testListStatus()
    {
        $this->asUser();
        $response = $this->get(route('enquiry-statuses.index'));

        $this->assertEquals(200, $response->status());
        $body = $response->getData(true);
        $this->assertEquals(0, count($body));

        EnquiryStatus::factory()->create(['name' => 'foo']);
        EnquiryStatus::factory()->create(['name' => 'baz']);
        EnquiryStatus::factory()->create(['name' => 'bar']);

        $response = $this->get(route('enquiry-statuses.index'));

        $body = $response->getData(true);
        $this->assertEquals(3, count($body));
    }

    public function testShowStatus()
    {
        $this->asUser();
        $status = EnquiryStatus::factory()->create([
            'name' => 'foo',
            'is_final' => true
        ]);

        $response = $this->get(route('enquiry-statuses.show', $status));

        $body = $response->getData();
        $this->assertEquals('foo', $body->name);
        $this->assertEquals(true, $body->is_final);
    }

    public function testAddStatus()
    {
        $this->asUser();
        $response = $this->postJson(route('enquiry-statuses.store'), [
            'name' => 'foo'
        ]);

        $this->assertEquals(201, $response->getStatusCode());
        $data = $response->getData();
        $status = EnquiryStatus::query()->find($data->id);

        $this->assertInstanceOf(EnquiryStatus::class, $status);
        $this->assertEquals($status->id, $data->id);
        $this->assertEquals('foo', $status->name);
    }

    public function testUpdateStatus()
    {
        $this->asUser();
        $status = EnquiryStatus::factory()->create(['name' => 'foo']);

        $response = $this->put(route('enquiry-statuses.update', $status->id), [
            'name' => 'baz-bar',
            'is_final' => false
        ]);

        $this->assertEquals(200, $response->getStatusCode());
        $status->refresh();

        $this->assertEquals('baz-bar', $status->name);
    }

    public function testRemoveStatus()
    {
        $this->asUser();
        $status = EnquiryStatus::factory()->create([
            'name' => 'foo',
        ]);

        $response = $this->delete(route('enquiry-statuses.destroy', $status));
        $this->assertEquals(200, $response->getStatusCode());

        $response = $this->get(route('enquiry-statuses.show', $status));
        $this->assertEquals(404, $response->getStatusCode());
    }

    public function testSeededEnquiryStatuses()
    {
        $this->seed(EnquiryStatusSeeder::class);

        $this->assertTrue(EnquiryStatus::query()->where('name', 'New')->exists());
        $this->assertTrue(EnquiryStatus::query()->where('name', 'Responded')->exists());
        $this->assertTrue(EnquiryStatus::query()->where('name', 'Waiting for feedback')->exists());
        $this->assertTrue(EnquiryStatus::query()->where('name', 'Closed')->exists());
    }

}
