<?php

namespace Mtc\Crm\Http\Controllers;

use Carbon\Carbon;
use Illuminate\Foundation\Validation\ValidatesRequests;
use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\Storage;
use Illuminate\Validation\ValidationException;
use Mtc\Crm\Contracts\EnquiryActionModel;
use Mtc\Crm\Contracts\EnquiryModel;
use Mtc\Crm\Http\Requests\EnquiryAddMessageRequest;
use Mtc\Crm\Http\Requests\EnquiryAssignRequest;
use Mtc\Crm\Http\Requests\EnquirySetStatusRequest;
use Mtc\Crm\Http\Requests\EnquirySubscribeRequest;
use Mtc\Crm\Http\Resources\EnquiryList;
use Mtc\Crm\Http\Resources\EnquiryViewResource;
use Mtc\Crm\Facades\Enquiries;
use Mtc\Crm\Facades\EnquiryData;
use Mtc\Crm\Models\Enquiry;

class EnquiryController
{
    use ValidatesRequests;

    /**
     * List enquiries
     *
     * @param Request $request
     * @return EnquiryList
     */
    public function index(Request $request): EnquiryList
    {
        $resource = Config::get('crm.enquiry_list_resource');
        return new $resource(Enquiries::showList($request));
    }

    /**
     * View enquiry  filters (types/statuses/tags)
     *
     * @return array
     */
    public function filters(): array
    {
        return [
            'types' => EnquiryData::getTypes(),
            'tags' => EnquiryData::getTags(),
            'statuses' => EnquiryData::getStatuses(),
        ];
    }

    /**
     * Show details of an enquiry
     *
     * @param int $id
     * @return EnquiryViewResource
     */
    public function details(int $id): EnquiryViewResource
    {
        $resource = Config::get('crm.enquiry_view_resource');
        return new $resource(Enquiries::find($id));
    }

    /**
     * Show enquiry (and possible statuses / users for interaction)
     *
     * @param int $enquiry_id
     * @return array
     */
    public function show(int $enquiry_id): array
    {
        return [
            'enquiry' => Enquiries::find($enquiry_id),
            'statuses' => EnquiryData::getStatuses(),
            'managers' => $this->usersToSubscribe(),
        ];
    }

    public function getFile($path)
    {
        $disk = Config::get('filesystems.file-storage');

        $filePath = urldecode($path);

        $enquiry = Enquiry::query()
            ->where('details', 'like', '%' . str_replace('/', '\\\\/', $filePath) . '%')
            ->first();

        // Check if the file exists
        if (!Storage::disk($disk)->exists($filePath) || !$enquiry) {
            abort(404, 'File not found');
        }

        $mimeType = Storage::disk($disk)->mimeType($filePath);

        return response()->file(Storage::disk($disk)->path($filePath), [
            'Content-Type' => $mimeType,
            'Content-Disposition' => 'inline; filename="' . basename($filePath) . '"'
        ]);
    }

    /**
     * Add a new message to enquiry
     *
     * @param EnquiryAddMessageRequest $request
     * @param int $enquiry_id
     * @return EnquiryViewResource
     */
    public function addMessage(EnquiryAddMessageRequest $request, int $enquiry_id): EnquiryViewResource
    {
        $enquiry = Enquiries::load($enquiry_id)
            ->addMessage(
                message: $request->input('message'),
                new_status_id: $request->input('new_status'),
                from_customer: false,
                user_id: Auth::id()
            );
        $resource = Config::get('crm.enquiry_view_resource');
        return new $resource($enquiry);
    }

    /**
     * Add a tag to enquiry
     *
     * @param int $enquiry_id
     * @param int $tag
     * @return bool
     */
    public function addTag(int $enquiry_id, int $tag): bool
    {
        return Enquiries::addTag($enquiry_id, $tag);
    }

    /**
     * Remove a tag from enquiry
     *
     * @param int $enquiry_id
     * @param int $tag
     * @return bool
     */
    public function removeTag(int $enquiry_id, int $tag): bool
    {
        return Enquiries::removeTag($enquiry_id, $tag);
    }

    /**
     * Update enquiry status
     *
     * @param EnquirySetStatusRequest $request
     * @param int $enquiry_id
     * @return EnquiryViewResource
     */
    public function setStatus(EnquirySetStatusRequest $request, int $enquiry_id): EnquiryViewResource
    {
        Enquiries::load($enquiry_id)->setStatus($request->input('status'));
        $resource = Config::get('crm.enquiry_view_resource');
        return new $resource(Enquiries::find($enquiry_id));
    }

    /**
     * Assign enquiry to user
     *
     * @param EnquiryAssignRequest $request
     * @param int $enquiry_id
     * @return EnquiryViewResource
     * @throws ValidationException
     */
    public function assign(EnquiryAssignRequest $request, int $enquiry_id): EnquiryViewResource
    {
        $user = App::make(Config::get('auth.providers.users.model'))
            ->newQuery()
            ->findOrFail($request->input('user'));
        Enquiries::load($enquiry_id)->assign($user, Auth::id());
        $resource = Config::get('crm.enquiry_view_resource');
        return new $resource(Enquiries::find($enquiry_id));
    }

    /**
     * Subscribe user to an enquiry
     *
     * @param EnquirySubscribeRequest $request
     * @param int $enquiry_id
     * @return EnquiryViewResource
     * @throws ValidationException
     */
    public function subscribe(EnquirySubscribeRequest $request, int $enquiry_id): EnquiryViewResource
    {
        $user = App::make(Config::get('auth.providers.users.model'))
            ->newQuery()
            ->findOrFail($request->input('subscriber'));
        Enquiries::load($enquiry_id)->subscribe($user);
        $resource = Config::get('crm.enquiry_view_resource');
        return new $resource(Enquiries::find($enquiry_id));
    }

    /**
     * Unsubscribe user(s) from an enquiry
     *
     * @param int $enquiry_id
     * @param int $user_id
     * @return EnquiryViewResource
     */
    public function unsubscribe(int $enquiry_id, int $user_id): EnquiryViewResource
    {
        $user = App::make(Config::get('auth.providers.users.model'))
            ->newQuery()
            ->findOrFail($user_id);
        Enquiries::load($enquiry_id)->unsubscribe($user);
        $resource = Config::get('crm.enquiry_view_resource');
        return new $resource(Enquiries::find($enquiry_id));
    }

    /**
     * View list of users that can be subscribed to an enquiry
     *
     * @return mixed
     */
    public function usersToSubscribe(): Collection
    {
        $in_tests = app()->runningUnitTests();
        return App::make(Config::get('auth.providers.users.model'))
            ->newQuery()
            ->when($in_tests === false, fn($query) => $query->role(Config::get('crm.role_for_crm_users')))
            ->select('id', 'name')
            ->get();
    }

    /**
     * Resubmit enquiry action to be performed once again
     *
     * @param EnquiryModel $enquiry
     * @param EnquiryActionModel $action
     * @return Response
     */
    public function resubmitAction(EnquiryModel $enquiry, EnquiryActionModel $action): Response
    {
        if (!empty($action->getAttribute('processed_at'))) {
            return response([
                'status' => 'ok',
                'processed_at' => $action->processed_at,
                'context' => 'Already Processed',
            ]);
        }

        $handler = Enquiries::makeAction($action->action_name);
        if ($handler->handle($action, $enquiry) === false) {
            $action->update([
                'failed_at' => Carbon::now(),
                'context' => 'Failed with reason: ' . $handler->failureReason()
            ]);
            return response([
                'status' => 'fail',
                'failed_at' => Carbon::now(),
                'context' => 'Failed with reason: ' . $handler->failureReason()
            ]);
        }

        $action->update([
            'processed_at' => Carbon::now(),
            'context' => $handler->successDetails(),
        ]);
        return response([
            'status' => 'ok',
            'processed_at' => Carbon::now(),
            'context' => $handler->successDetails(),
        ]);
    }
}
